#include "BME280.h"

BME280 bme ;

// ---------------- I2C 基本関数 ----------------
/**
 * @brief １バイト読み込み
 * @param reg レジスタアドレス
 * @return 読み込んだデータ
 */
uint8_t BME280::read8(uint8_t reg) {
  Wire.beginTransmission(BME280_ADDR);
  Wire.write(reg);
  Wire.endTransmission();
  Wire.requestFrom(BME280_ADDR, (uint8_t)1);
  return Wire.read();
}

/**
 * @brief ２バイト読み込み
 * @param reg レジスタアドレス
 * @return 読み込んだデータ
 * 
 */
uint16_t BME280::read16(uint8_t reg) {
  Wire.beginTransmission(BME280_ADDR);
  Wire.write(reg);
  Wire.endTransmission();
  Wire.requestFrom(BME280_ADDR, (uint8_t)2);
  uint16_t value = Wire.read();
  value |= Wire.read() << 8;
  return value;
}

/**
 * @brief ２バイト読み込み（符号付き）
 * @param reg レジスタアドレス  
 * @return 読み込んだデータ
 * 
 */
int16_t BME280::readS16(uint8_t reg) {
  return (int16_t)read16(reg);
}

/**
 * @brief １バイト書き込み
 * @param reg レジスタアドレス
 * @param val 書き込むデータ
 *  
 */
void BME280::write8(uint8_t reg, uint8_t val) {
  Wire.beginTransmission(BME280_ADDR);
  Wire.write(reg);
  Wire.write(val);
  Wire.endTransmission();
}

// ---------------- キャリブレーション読込 ----------------
/**
 * @brief キャリブレーションデータ読み込み
 * 
 */
void BME280::readCalibrationData() {
  dig_T1 = read16(0x88);
  dig_T2 = readS16(0x8A);
  dig_T3 = readS16(0x8C);

  dig_P1 = read16(0x8E);
  dig_P2 = readS16(0x90);
  dig_P3 = readS16(0x92);
  dig_P4 = readS16(0x94);
  dig_P5 = readS16(0x96);
  dig_P6 = readS16(0x98);
  dig_P7 = readS16(0x9A);
  dig_P8 = readS16(0x9C);
  dig_P9 = readS16(0x9E);

  dig_H1 = read8(0xA1);
  dig_H2 = readS16(0xE1);
  dig_H3 = read8(0xE3);

  uint8_t e4 = read8(0xE4);
  uint8_t e5 = read8(0xE5);
  uint8_t e6 = read8(0xE6);

  dig_H4 = (e4 << 4) | (e5 & 0x0F);
  dig_H5 = (e6 << 4) | (e5 >> 4);
  dig_H6 = (int8_t)read8(0xE7);
}

// ---------------- 生データ取得 ----------------
/**
 * @brief 生データ読み込み
 * @param adc_T 温度生データ格納先  
 * @param adc_P 気圧生データ格納先
 * @param adc_H 湿度生データ格納先
 * 
 */
void BME280::readRawData(int32_t &adc_T, int32_t &adc_P, int32_t &adc_H) {
  Wire.beginTransmission(BME280_ADDR);
  Wire.write(0xF7);
  Wire.endTransmission();
  Wire.requestFrom(BME280_ADDR, (uint8_t)8);

  adc_P = ((uint32_t)Wire.read() << 12) |
          ((uint32_t)Wire.read() << 4)  |
          (Wire.read() >> 4);

  adc_T = ((uint32_t)Wire.read() << 12) |
          ((uint32_t)Wire.read() << 4)  |
          (Wire.read() >> 4);

  adc_H = ((uint32_t)Wire.read() << 8) | Wire.read();
}

// ---------------- 補正計算 ----------------
/**
 * @brief 温度補正計算
 * @param adc_T 温度生データ
 * @return 補正後温度（℃）
 * 
 */
float BME280::compensateTemperature(int32_t adc_T) {
  int32_t var1, var2;
  var1 = ((((adc_T >> 3) - ((int32_t)dig_T1 << 1))) * dig_T2) >> 11;
  var2 = (((((adc_T >> 4) - (int32_t)dig_T1) *
            ((adc_T >> 4) - (int32_t)dig_T1)) >> 12) *
          dig_T3) >> 14;
  t_fine = var1 + var2;
  return ((t_fine * 5 + 128) >> 8) / 100.0  ;
}

/**
 * @brief 気圧補正計算
 * @param adc_P 気圧生データ
 * @return 補正後気圧（hPa）
 * 
 */
float BME280::compensatePressure(int32_t adc_P) {
  int64_t var1, var2, p;
  var1 = (int64_t)t_fine - 128000;
  var2 = var1 * var1 * dig_P6;
  var2 += (var1 * dig_P5) << 17;
  var2 += ((int64_t)dig_P4) << 35;
  var1 = ((var1 * var1 * dig_P3) >> 8) +
         ((var1 * dig_P2) << 12);
  var1 = (((((int64_t)1) << 47) + var1) * dig_P1) >> 33;

  if (var1 == 0) return 0;

  p = 1048576 - adc_P;
  p = (((p << 31) - var2) * 3125) / var1;
  var1 = (dig_P9 * (p >> 13) * (p >> 13)) >> 25;
  var2 = (dig_P8 * p) >> 19;
  p = ((p + var1 + var2) >> 8) + ((int64_t)dig_P7 << 4);
  return (p >> 8) / 100.0;
}

/**
 * @brief 湿度補正計算
 * @param adc_H 湿度生データ
 * @return 補正後湿度（％）
 * 
 */
float BME280::compensateHumidity(int32_t adc_H) {
  int32_t v_x1;
  v_x1 = t_fine - 76800;
  v_x1 = (((((adc_H << 14) - (dig_H4 << 20) -
              (dig_H5 * v_x1)) + 16384) >> 15) *
           (((((((v_x1 * dig_H6) >> 10) *
               (((v_x1 * dig_H3) >> 11) + 32768)) >> 10) +
             2097152) * dig_H2 + 8192) >> 14));
  v_x1 -= (((((v_x1 >> 15) * (v_x1 >> 15)) >> 7) * dig_H1) >> 4);
  v_x1 = (v_x1 < 0) ? 0 : v_x1;
  v_x1 = (v_x1 > 419430400) ? 419430400 : v_x1;
  return (v_x1 >> 12) / 1024.0;
}

/**
 * @brief BME280初期化
 * 
 */
void BME280::initBME280() {
  // 設定（通常モード）
  write8(0xF2, 0x01); // 湿度 oversampling x1
  write8(0xF4, 0x27); // 温度/気圧 oversampling x1, normal mode
  write8(0xF5, 0xA0); // standby 1000ms
  readCalibrationData();
}

/**
 * @brief 補正済み環境値取得
 * @param temp 温度格納先（℃）
 * @param pres 気圧格納先（hPa）
 * @param hum 湿度格納先（％）
 * 
 */
void BME280::getCompensate(float &temp , float &pres , float &hum) {
  int32_t adc_T, adc_P, adc_H;
  readRawData(adc_T, adc_P, adc_H);

  temp = compensateTemperature(adc_T) + TWERIT_DEFF_T ; // TWELITEの計測値との差を補正
  pres = compensatePressure(adc_P);
  hum  = compensateHumidity(adc_H) + TWERIT_DEFF_H ; // TWELITEの計測値との差を補正
  //湿度の範囲補正
  if (hum >= 100.0) { hum = 99.9 ; }
}
