/*
 * SSD1306/SSD1315に７セグメントっぽいフォントで表示用のクラスを流用
 * HX8357用にビットマップデータを256色用に変換して返す
 * 
 */
#include <Arduino.h>
#include "string.h"
#include "bitmapdata.h"
#include "DISP7SEG.h"
#include "HX8357.h"
#include "debug.h"

DISP7SEG disp7seg ;

// 7セグメント調フォント表示
/**
 * @brief ７セグメントフォントビットマップ取得
 * @param num 表示する文字（0-9,-,.,:）
 * @param colorF 前景色
 * @param colorB 背景色
 * @return ビットマップデータポインタ
 * 
 */
uint16_t * DISP7SEG::disp7SEG(int num,uint16_t colorF,uint16_t colorB) {
  int segW = base7SEG_W ;
  if (num <= 10) {
    if (num < 0) { num = 10 ; }
    bool flag7[7] ;
    for (int i=0;i<7;i++) {
      flag7[i] = addMap[num][i] ; 
    }
    for (int i = 0; i < 16*4; i++) {
        bitMapBuff[i] = 0 ;
      if (flag7[0]) { bitMapBuff[i] |= SEGa[i] ; }  // a
      if (flag7[1]) { bitMapBuff[i] |= SEGb[i] ; }  // b
      if (flag7[2]) { bitMapBuff[i] |= SEGc[i] ; }  // c
      if (flag7[3]) { bitMapBuff[i] |= SEGd[i] ; }  // d
      if (flag7[4]) { bitMapBuff[i] |= SEGe[i] ; }  // e
      if (flag7[5]) { bitMapBuff[i] |= SEGf[i] ; }  // f
      if (flag7[6]) { bitMapBuff[i] |= SEGg[i] ; }  // g
    }
  } else {
    uint8_t *bitMap = NULL ;
    switch(num) {
      case 11: bitMap = SEGdot ; break ; // DOT
      case 12: bitMap = SEGcoron ; break ; // ：
      case 13: bitMap = SEGtemp ; break ; // 温度計
      case 14: bitMap = SEGoc ; break ; // ℃
      case 15: bitMap = SEGhum ; break ; // 湿度計
      case 16: bitMap = SEGp ; break ; // ％
      case 17: bitMap = SEGPress ; break ; // 気圧
      case 18: bitMap = SEGhpa ; segW = SEGhpa_W ; break ; // hPa
      case 19: bitMap = SEGPress ; break ; // 照度
      case 20: bitMap = SEGhpa ; segW = SEGhpa_W ; break ; //Lx
      default: bitMap = NULL ;
    }
    if (bitMap != NULL) {
      for (int i = 0; i < segW * 4; i++) {
        bitMapBuff[i] = bitMap[i] ; 
      } 
    } else {
      memset(bitMapBuff,0,sizeof(bitMapBuff)) ;
    }
  }

  // --- 256色用に変換 --- 
  const uint8_t* mask = mcbp;   // ビットマスク参照
  uint16_t* dst = bitMap7Seg;
  for (int y = 0; y < base7SEG_H; y++) {
    int row = y >> 3;         // y / 8
    uint8_t bit = mask[y & 7];
    const uint8_t* src = &bitMapBuff[row * segW];

    for (int x = 0; x < segW; x++) {
      *dst++ = (src[x] & bit) ? colorF : colorB;
    }
 }  
  
  return bitMap7Seg ;
}

/**
 * @brief 文字を7セグメント用の数値に変換
 * @param c 変換する文字
 * @return 変換後数値（0-9,-:10,.:11,:,12）
 * 
 */
int DISP7SEG::char2num(char c) {
  int num = 99 ;
  switch(c) {
    case '-': num = 10 ; break ;
    case '.': num = 11 ; break ;
    case ':': num = 12 ; break ;
    case '0': num = 0 ; break ;
    case '1': num = 1 ; break ;
    case '2': num = 2 ; break ;
    case '3': num = 3 ; break ;
    case '4': num = 4 ; break ;
    case '5': num = 5 ; break ;
    case '6': num = 6 ; break ;
    case '7': num = 7 ; break ;
    case '8': num = 8 ; break ;
    case '9': num = 9 ; break ;
    default: num = 99 ; break ;
  }
  return num ;
}

/**
 * @brief 日時表示
 * @param x 開始X座標
 * @param y 開始Y座標
 * @param year 年
 * @param mon 月
 * @param mday 日
 * @param hour 時
 * @param tm_min 分
 * @param tm_sec 秒
 * @param colorF 前景色
 * @param colorB 背景色
 * 
 */
void DISP7SEG::dispTime(int x, int y ,int year ,int mon ,int mday ,
                        int hour ,int tm_min ,int tm_sec ,uint16_t colorF ,uint16_t colorB) {
  x = dispDate(x,y,year,mon,mday,colorF,colorB) ;
  dispTime(x+16,y,hour,tm_min,tm_sec,colorF,colorB) ;
}

/**
 * @brief 日付表示
 * @param x 開始X座標
 * @param y 開始Y座標
 * @param year 年
 * @param mon 月
 * @param mday 日
 * @param colorF 前景色
 * @param colorB 背景色
 * @return 次の表示X座標
 * 
 */
int DISP7SEG::dispDate(int x, int y ,int year ,int mon ,int mday ,uint16_t colorF ,uint16_t colorB) {
  int num = 10 ;
  int numLen = 0 ;
  uint16_t *font ;

  sprintf(strBuff,"%04d-%02d-%02d",year,mon,mday) ;
  numLen = strlen(strBuff) ;
  for (int i=0;i<numLen;i++) {
    num = char2num(strBuff[i]) ;
    font = disp7SEG(num,colorF,colorB) ;
    tft.drowBitmap(x , y , base7SEG_W , base7SEG_H , font);
    x += base7SEG_W ;
  }

  return x ;
}

/**
 * @brief 時刻表示
 * @param x 開始X座標
 * @param y 開始Y座標
 * @param hour 時
 * @param tm_min 分
 * @param tm_sec 秒
 * @param colorF 前景色
 * @param colorB 背景色
 * @return 次の表示X座標
 * 
 */
int DISP7SEG::dispTime(int x, int y , int hour , int tm_min , int tm_sec , uint16_t colorF,uint16_t colorB) {
  int num = 10 ;
  int numLen = 0 ;
  uint16_t *font ;

  if (tm_sec < 0) {
    sprintf(strBuff,"%02d:%02d",hour,tm_min) ;
  } else {
    sprintf(strBuff,"%02d:%02d:%02d",hour,tm_min,tm_sec) ;
  }
  numLen = strlen(strBuff) ;
  for (int i=0;i<numLen;i++) {
    num = char2num(strBuff[i]) ;
    font = disp7SEG(num,colorF,colorB) ;
    tft.drowBitmap(x , y , base7SEG_W , base7SEG_H , font);
    x += base7SEG_W ;
  }

  return x ;
}

/**
 * CONFIGアイコン表示::HX8357用に作成した画像データを使用する
 * @brief アイコン表示
 * @param x 開始X座標
 * @param y 開始Y座標
 * @param iconNo アイコン番号
 * @param colorF 前景色
 * @param colorB 背景色
 * 
 */
void DISP7SEG::dispIcon(int x, int y , int iconNo , uint16_t colorF,uint16_t colorB) {
  uint8_t *bitMap = NULL ;
  int w = 0 ;
  int h = 0 ;
  switch(iconNo) {
    case 1: // WiFi
      bitMap = imageWiFi ;
      w = imageWiFi_W;
      h = imageWiFi_H;
      break ;

    case 2: // CONFIG
      bitMap = imageCONFIG ;
      w = imageCONFIG_W;
      h = imageCONFIG_H;
      break ;

    case 3: // ALARM
      bitMap = imageALARM ;
      w = imageALARM_W;
      h = imageALARM_H;
      break ;

    case 4: // LAUNDRY
      bitMap = imageLaundry ;
      w = imageLaundry_W;
      h = imageLaundry_H;
      break ;

    case 5: // AirPressure
      bitMap = SEGPress ;
      w = SEGPress_W;
      h = SEGPress_H;
      break ;

  }

  if (bitMap != NULL) {
    tft.drawXBitmap(x, y, w, h, bitMap, colorF, colorB) ;
  }
}

/**
 * @brief 補正済み環境値表示
 * @param x 開始X座標
 * @param y 開始Y座標
 * @param statType 表示する環境値種別 0:気温 1:湿度 2:気圧
 * @param stat 環境値
 * @param colorF 前景色
 * @param colorB 背景色
 * @return 次の表示X座標
 * 
 */
int DISP7SEG::dispCompensate(int x, int y , int statType , float stat , uint16_t colorF , uint16_t colorB){
  int num = 99 ;
  int numLen = 0 ;
  uint16_t *font ;

  switch(statType) {
    case 0: // 気温
      sprintf(strBuff,"%4.1f",stat) ;
      DPRINTF("気温:%s\n",strBuff) ;
      numLen = strlen(strBuff) ;
      for (int i=0;i<numLen;i++) {
        num = char2num(strBuff[i]) ;
        font = disp7SEG(num,colorF,colorB) ;
        tft.drowBitmap(x , y , base7SEG_W , base7SEG_H , font);
        x += (strBuff[i] == '.')?9:base7SEG_W ;
      }
      //--
      font = disp7SEG(14,colorF,colorB) ;
      tft.drowBitmap(x , y , base7SEG_W , base7SEG_H , font);
      x += base7SEG_W ;
      break ;

    case 1: // 湿度
      sprintf(strBuff,"%4.1f",stat) ;
      DPRINTF("湿度:%s\n",strBuff) ;
      numLen = strlen(strBuff) ;
      for (int i=0;i<numLen;i++) {
        num = char2num(strBuff[i]) ;
        font = disp7SEG(num,colorF,colorB) ;
        tft.drowBitmap(x , y , base7SEG_W , base7SEG_H , font);
        x += (strBuff[i] == '.')?9:base7SEG_W ;
      }
      //--
      font = disp7SEG(16,colorF,colorB) ;
      tft.drowBitmap(x , y , base7SEG_W , base7SEG_H , font);
      x += base7SEG_W ;
      break ;

    case 2: // 気圧
      sprintf(strBuff,"%4d",(int)stat) ;
      DPRINTF("気圧:%s\n",strBuff) ;
      numLen = strlen(strBuff) ;
      for (int i=0;i<numLen;i++) {
        num = char2num(strBuff[i]) ;
        font = disp7SEG(num,colorF,colorB) ;
        tft.drowBitmap(x , y , base7SEG_W , base7SEG_H , font);
        x += (strBuff[i] == '.')?9:base7SEG_W ;
      }
      //--
      font = disp7SEG(18,colorF,colorB) ;
      tft.drowBitmap(x , y , SEGhpa_W , base7SEG_H , font);
      x += SEGhpa_W ;
      break ;

    default: // 整数値
      sprintf(strBuff,"%d",(int)stat) ;
      numLen = strlen(strBuff) ;
      for (int i=0;i<numLen;i++) {
        num = char2num(strBuff[i]) ;
        font = disp7SEG(num,colorF,colorB) ;
        tft.drowBitmap(x , y , base7SEG_W , base7SEG_H , font);
        x += base7SEG_W ;
      }
      break ;

  }
  DPRINTF("X=%d\n",x) ;
  return x ;
}
