/*
 * SDカードから東雲フォントを読み出して表示するクラス
 * 
 */
#include <Arduino.h>
#include <FS.h>
#include <SD.h>
#include "FontSD.h"
#include "debug.h"

FontSD fontSD ;

// UTF-16半角カナの範囲
#define UTF16_HANKAKU_START 0xFF61
#define UTF16_HANKAKU_END   0xFF9F

// JIS X 0201 の対応開始（UTF-16 U+FF61 ⇔ JIS 0xA1）
#define JIS_HANKAKU_START 0xA1

/**
 * @brief SDカードからフォントファイルを開く
 * @return true:成功 false:失敗
 * 
 */
bool FontSD::openSD() {
    if (!SD.exists(FONT_FILE)) {
        return false ;
    }  

    fontFile = SD.open(FONT_FILE, FILE_READ) ;
    if (fontFile) {
        return true ;
    }
    return false ;
}

/**
 * @brief フォントファイルを閉じる
 * 
 */
void FontSD::closeSD() {
    fontFile.close( );
}

/**
 * @brief UTF-8文字列のバイト長を取得
 * @param utf8codes UTF-8文字列ポインタ
 * @return 文字長（バイト数）
 *          UTF8の文字長(2 or 3)を返す
 *          ASCIIの場合は1を返す
 * 
 */
int FontSD::UTF8LEN(uint8_t *utf8codes) {
  int utfLen = 0 ;
  if( utf8codes[0] < 0x80 ){
    utfLen = 1 ;
  } else
  if( utf8codes[0] < 0xE0 ){
    utfLen = 2 ;
  } else {
    utfLen = 3 ;
  }
  return utfLen ;
}

/**
 * @brief UTF-16半角カナをJIS X 0201に変換
 * @param utf16_char UTF-16半角カナ文字 
 * @return JIS X 0201コード（変換不可の場合は0を返す）
 * 
 */
uint8_t FontSD::utf16_to_jis_kana(uint16_t utf16_char) {
    if (utf16_char == 0x3000) {
      return 0x20 ; //  全角スペースを半角スペースに変換
    }
    if (utf16_char >= UTF16_HANKAKU_START && utf16_char <= UTF16_HANKAKU_END) {
        return (uint8_t)(utf16_char - UTF16_HANKAKU_START + JIS_HANKAKU_START);
    } else {
        // 範囲外（例: 全角やその他の記号など）
        return 0; // またはエラーコード（例: 0x00）
    }
}

// UTF8->UTF16
/**
 * @brief UTF-8文字列をUTF-16に変換
 * @param utf8codes UTF-8文字列ポインタ
 * @return UTF-16コード
 * 
 */
uint16_t FontSD::UTF8ToUTF16(uint8_t *utf8codes) {
  uint16_t utf16Code = 0x0000 ;
  int codeLen = UTF8LEN(utf8codes) ;
 
  if (codeLen == 2) {
    ((uint8_t *)&utf16Code)[0] = ((utf8codes[0] & 0x1F)<<6) + (utf8codes[1] & 0x3F);
    ((uint8_t *)&utf16Code)[1] = ((utf8codes[0] & 0x1F)>>2);
  } else
  if (codeLen == 3) {
    ((uint8_t *)&utf16Code)[1] = ((utf8codes[0] & 0xF)<<4) + ((utf8codes[1] & 0x3F)>>2);
    ((uint8_t *)&utf16Code)[0] = ((utf8codes[1] & 0x3F)<<6) + (utf8codes[2] & 0x3F);
  }
  return utf16Code ;
}

/**
 * @brief 16x16フォントデータ読み出し
 * @param code 文字コード
 * @return フォントデータポインタ
 * 
 */
uint8_t* FontSD::readFont16x16(uint16_t code) {
    int pos = code * 32 + 256 * 16 ;
    fontFile.seek(pos) ;
    memset(fontdata,0,FONT_SIZE_F) ;
    for (int i=0;i<FONT_SIZE_F;i++) {
#if BITREVERS
      fontdata[i] = fontFile.read( );
#else
      fontdata[i] = fontSD.reverse_bits((uint8_t)fontFile.read( )) ;
#endif      
    }
    return fontdata ;
}

/**
 * @brief 8x16フォントデータ読み出し
 * @param code 文字コード
 * @return フォントデータポインタ
 * 
 */
uint8_t* FontSD::readFont8x16(uint8_t code) {
    int pos = code * 16 ;
    fontFile.seek(pos) ;
    memset(fontdata,0,FONT_SIZE_F) ;
    for (int i=0;i<FONT_SIZE_H;i++) {
#if BITREVERS
      fontdata[i] = fontFile.read( );
#else
      fontdata[i] = fontSD.reverse_bits((uint8_t)fontFile.read( )) ;
#endif      
    }
    return fontdata ;
}

// 8ビットのビット順を逆にする関数
/**
 * @brief 8ビットのビット順を逆にする
 * @param value 元の8ビット値
 * @return ビット順を逆にした8ビット値
 * 
 */
uint8_t FontSD::reverse_bits(uint8_t value) {
    uint8_t result = 0;
    for (int i = 0; i < 8; ++i) {
        result <<= 1;           // 左に1ビットシフト
        result |= (value & 1);  // 最下位ビットを取り出して追加
        value >>= 1;            // 元の値を右にシフト
    }
    return result;
}

// 文字列を表示
/**
 * @brief 32x16フォントで文字を表示
 * @param code 文字コード
 * @param color 表示色
 * 
 */
void FontSD::disp32G(uint16_t code, uint16_t colorF, uint16_t colorB , bool blankSkip, bool doubleSize) {
    // 先頭の空白は描画しない
    if (blankSkip && curX == curTop && (code == 0x0020 || code == 0x3000)) {
      return ;
    }
    uint8_t *fontData = fontSD.readFont16x16(code) ;

#if !BITREVERS
    for (int i=0;i<FONT_SIZE_F;i++) {
        fontData[i] = fontSD.reverse_bits(fontData[i]) ;
    }
#endif

    tft.drawXBitmap(curX, curY, FONT32WIDTH, FONT_HIGHT, fontData , colorF , colorB , doubleSize);
    curX += (doubleSize)?FONT32WIDTH*2:FONT32WIDTH ;
    if (curX >= curTop+curW) {
        curX = curTop ;
        curY += (doubleSize)?FONT_HIGHT*2:FONT_HIGHT ;
    }
}

/**
 * @brief 16x16フォントで文字を表示
 * @param code 文字コード
 * @param color 表示色
 * 
 */
void FontSD::disp16G(uint8_t code, uint16_t colorF , uint16_t colorB , bool blankSkip, bool doubleSize) {
    // 先頭の空白は描画しない
    if (blankSkip && curX == curTop && code == 0x20) {
      return ;
    }

    uint8_t *fontData = fontSD.readFont8x16(code) ;

#if !BITREVERS
    for (int i=0;i<FONT_SIZE_H;i++) {
        fontData[i] = fontSD.reverse_bits(fontData[i]) ;
    }
#endif

    tft.drawXBitmap(curX, curY, FONT16WIDTH, FONT_HIGHT, fontData , colorF , colorB , doubleSize);
    curX += (doubleSize)?FONT16WIDTH*2:FONT16WIDTH ;
    if (curX >= curTop+curW) {
        curX = curTop ;
        curY += (doubleSize)?FONT_HIGHT*2:FONT_HIGHT ;
   }
}

/**
 * @brief 文字列を表示
 * @param x 表示X座標
 * @param y 表示Y座標
 * @param w 表示幅
 * @param text 表示文字列
 * @param color 文字色
 * @return true:成功 false:失敗
 * 
 */
bool FontSD::dispTextG(int x, int y , int w , char *text , uint16_t colorF, uint16_t colorB, bool blankSkip, bool doubleSize) {
    char str[8] ;
    int cPos = 0 ;
    curX = x ;
    curY = y ;
    curW = w ;
    curTop = x ;

    DPRINTF("dispTextG:%s\n",text) ;
    if (!openSD()) {
      DPRINTLN("dispTextG:SD Card fale.");
      return false ;
    }

    while(text[cPos] != 0) {
        int cLen = fontSD.UTF8LEN((uint8_t *)&text[cPos]) ;
        uint16_t code = 0 ;
        uint8_t code8 = 0 ;
        if (cLen > 1) {
            code = fontSD.UTF8ToUTF16((uint8_t *)&text[cPos]); 
            code8 = fontSD.utf16_to_jis_kana(code);
        } else {
            code8 = (uint16_t)text[cPos] ;
        }
        for (int b=0;b<cLen;b++) {
            str[b] = text[cPos+b] ;
        }
        str[cLen] = 0 ;

        if (code8 == 0) {
            disp32G(code,colorF,colorB,blankSkip,doubleSize) ;
        } else {
            disp16G(code8,colorF,colorB,blankSkip,doubleSize) ;
        }
        cPos += cLen ;
    }
    closeSD( ) ;
    return true ;
}
