/**
 * TFT HX8357 Driver
 * 
 */
#include <FS.h>
#include <SD.h>
#include "HX8357.h"
#include "debug.h"

HX8357 tft ;

/**
 * @brief TFT コントローラの初期化を行う
 *
 * SPI の初期化やコントローラに対する初期コマンド列の送信を行い、
 * ディスプレイを表示可能な状態にします。
 */
void HX8357::initTFT(int csPin , int dcPin) {
  _cs = csPin ;
  _dc = dcPin ;

  colorFG = COLOR_WHITE ; // ディフォルト前景色
  colorBG = COLOR_BLACK ; // ディフォルト背景色

  // HX8357 / HX8357D 初期化シーケンスデータ
  static uint8_t initd[] = {
    0x01 ,                 // SWRESET: ソフトウェアリセット
    0x80 + 100/5,          // 10ms 待機（0x80 + n*5ms の形式）

    0xB9 ,                 // SETC: 拡張コマンドセット有効化
    3,
        0xFF,              // 拡張コマンド有効
        0x83,              // 識別コード
        0x57,              // 識別コード（IC ID）
    0xFF,
    0x80 + 500/5,          // 300ms 待機（電源安定化のため）

    0xB3 ,                 // SETRGB: RGBインターフェイス設定
    4,
        0x80,              // SDO 出力有効 (0x80=有効, 0x00=無効)
        0x00,              // RGBフォーマット設定
        0x06,              // ドライブ設定
        0x06,              // ドライブ設定

    0xB6 ,                 // SETCOM: VCOM 電圧設定
    1,
        0x25,              // VCOM = 約 -1.52V

    0xB0 ,                 // SETOSC: オシレータ（フレームレート）設定
    1,
        0x68,              // 通常モード約70Hz、アイドルで約55Hz

    0xCC ,                 // SETPANEL: パネル設定
    1,
        0x05,              // BGRモード、Gate方向反転

    0xB1 ,                 // SETPWR1: 電源制御
    6,
        0x00,              // 通常動作（Deep Standby 無効）
        0x15,              // BT（電圧設定）
        0x1C,              // VSPR（正電源設定）
        0x1C,              // VSNR（負電源設定）
        0x83,              // AP（振幅設定）
        0xAA,              // FS（ドライブ周波数設定）

    0xC0 ,                 // SETSTBA: Source オプション設定
    6,
        0x50,              // OPON 通常
        0x50,              // OPON アイドル
        0x01,              // STBA タイミング
        0x3C,              // STBA タイミング
        0x1E,              // STBA タイミング
        0x08,              // GEN（駆動強度）

    0xB4 ,                 // SETCYC: パネル駆動サイクル設定
    7,
        0x02,              // NW = 0x02 ライン数
        0x40,              // RTN = フロントポーチ
        0x00,              // DIV = クロック分周
        0x2A,              // DUM = バックポーチ
        0x2A,              // DUM = バックポーチ2
        0x0D,              // GDON = Gate On 時間
        0x78,              // GDOFF = Gate Off 時間

    0xE0 ,                 // SETGAMMA: ガンマカーブ設定
    34,                    // 34個のパラメータ
        // +ガンマ
        0x02,0x0A,0x11,0x1d,0x23,
        0x35,0x41,0x4b,0x4b,0x42,
        0x3A,0x27,0x1B,0x08,0x09,
        0x03,
        // -ガンマ
        0x02,0x0A,0x11,0x1d,0x23,
        0x35,0x41,0x4b,0x4b,0x42,
        0x3A,0x27,0x1B,0x08,0x09,
        0x03,
        0x00,0x01,          // 終端パラメータ

    0x3A ,                 // COLMOD: カラーモード
    1,
        0x55,              // 16bit (RGB565)

    0x36 ,                 // MADCTL: メモリアクセス制御
    1,
        0x00,              // BGR・標準縦画面

    0x35 ,                 // TEON: テアリング制御
    1,
        0x00,              // Tearing OFF（VSYNC のみ）

    0x44 ,                 // TEARLINE: テアリングライン設定
    2,
        0x00,
        0x02,              // ライン 2 を指定

    0x11 ,                 // SLPOUT: スリープ解除
    0x80 + 150/5,          // 150ms 待機

    0x29 ,                 // DISPON: 表示 ON
    0x80 + 50/5,           // 50ms 待機

    0,                     // 終端 (0 = コマンドリスト終了)
  };

  _tft_w = TFT_WIDTH ;
  _tft_h = TFT_HIGHT ;

  // ピン初期化
  pinMode(_cs, OUTPUT);
  pinMode(_dc, OUTPUT);
  digitalWrite(_cs, HIGH);
  digitalWrite(_dc, HIGH);

  // SPI 初期化
  SPI.begin(PIN_CLK,PIN_MISO,PIN_MOSI,-1);  // SPI.begin(SCK, MISO, MOSI, SS);

  // --- 初期化シーケンス ---
  // ソフトリセット（任意）
  writeCommand(0x01); // SWRESET
  delay(150);

  uint8_t cmd,nByte,buff[48] ;
  int dpos = 0 ;

  // Sleep Out
  writeCommand(0x11); // SLPOUT
  delay(120);
  
  while (initd[dpos] != 0) {
    cmd = initd[dpos] ; dpos ++ ;
    writeCommand(cmd) ;
    nByte = initd[dpos] ; dpos ++ ;
    if ((nByte & 0x80) != 0) {
      int wait = nByte & 0x7F ;
      delay(wait) ;
    } else {
      uint8_t *addr = &initd[dpos] ;
      dpos += nByte ;
      memcpy(buff,addr,nByte) ;
      writeDataBuffer(buff, nByte);
    }
  }

  tftDeselect();
}

/**
 * @brief 単一のコマンドバイトを送信する
 *
 * @param cmd 送信するコマンドバイト
 */
void HX8357::writeCommand(uint8_t cmd) {
  SPI.beginTransaction(hxSpiSettings);
  commandMode() ;
  tftSelect();
  SPI.transfer(cmd);
  tftDeselect();
  SPI.endTransaction();
}

/**
 * @brief 単一のデータバイトを送信する
 *
 * データ/コマンドピンをデータモードにして 1 バイト送信します。
 *
 * @param data 送信するデータバイト
 */
void HX8357::writeData(uint8_t data) {
  SPI.beginTransaction(hxSpiSettings);
  dataMode( ) ;
  tftSelect();
  SPI.transfer(data);
  tftDeselect();
  SPI.endTransaction();
}

/**
 * @brief 単一の16ビットデータを送信する
 *
 * データ/コマンドピンをデータモードにして 2 バイト送信します。
 *
 * @param data 送信する16ビットデータ
 */
void HX8357::writeData16(uint16_t data) {
  SPI.beginTransaction(hxSpiSettings);
  dataMode( ) ;
  tftSelect();
  SPI.transfer16(data);
  tftDeselect();
  SPI.endTransaction();
}

/**
 * @brief データバッファを連続送信する
 *
 * バッファ内の `len` バイトを連続で SPI 送信します。長いデータ列の
 * 転送に使用します。
 *
 * @param buf 送信するデータバッファへのポインタ
 * @param len 送信するバイト数
 */
void HX8357::writeDataBuffer(const uint8_t *buf, size_t len) {
  if (len == 0) return;
  SPI.beginTransaction(hxSpiSettings);
  dataMode( ) ;
  tftSelect();
  // 大きいバッファなら chunk 化するのも可
  for (size_t i = 0; i < len; ++i) {
    SPI.transfer(buf[i]);
  }
  tftDeselect();
  SPI.endTransaction();
}

/**
 * @brief 16ビットデータバッファを連続送信する
 *
 * バッファ内の `len` 個の 16ビットデータを連続で SPI 送信します。長いデータ列の
 * 転送に使用します。
 *
 * @param buf 送信する16ビットデータバッファへのポインタ
 * @param len 送信する16ビットデータ数
 */
void HX8357::writeDataBuffer16(const uint16_t *buf, size_t len) {
  if (len == 0) return;
  SPI.beginTransaction(hxSpiSettings);
  dataMode( ) ;
  tftSelect();
  // 大きいバッファなら chunk 化するのも可
  for (size_t i = 0; i < len; ++i) {
    SPI.transfer16(buf[i]);
  }
  tftDeselect();
  SPI.endTransaction();
}

/**
 * Pixcel描画範囲を設定する
 * 
 * @param x,y 開始位置
 * @param w,h 画像サイズ 幅、高さ
 */
void HX8357::setWindow(uint16_t x , uint16_t y , uint16_t w , uint16_t h) {
  uint16_t xe = x + w -1 ;
  uint16_t ye = y + h -1 ;

  SPI.beginTransaction(hxSpiSettings);
  commandMode() ;
  tftSelect();
  SPI.transfer(0x2A);
  dataMode( ) ;
  SPI.transfer16(x);
  SPI.transfer16(xe);

  commandMode() ;
  SPI.transfer(0x2B);
  dataMode( ) ;
  SPI.transfer16(y);
  SPI.transfer16(ye);

  tftDeselect();
  SPI.endTransaction();
}

/**
 * @brief 表示方向を設定する
 * 
 * @param dir 0-ノーマル 1-90° 2-180° 3-270°
 */
void HX8357::setDir(int dir) {
  // bit7 - Row Address Order - 上下反転
  // bit6 - Column Address Order - 左右反転
  // bit5 - Row/Column Exchange - 転置＝90°回転
  // bit4 - Vertical Refresh Order - 行スキャン方向

  uint8_t data = 0x00 ;
  switch(dir) {
    case 0:data = 0x00 ; _tft_w = TFT_WIDTH ; _tft_h = TFT_HIGHT ; break ;
    case 1:data = 0x60 ; _tft_h = TFT_WIDTH ; _tft_w = TFT_HIGHT ; break ;
    case 2:data = 0xC0 ; _tft_w = TFT_WIDTH ; _tft_h = TFT_HIGHT ; break ;
    case 3:data = 0xA0 ; _tft_h = TFT_WIDTH ; _tft_w = TFT_HIGHT ; break ;
  }

  SPI.beginTransaction(hxSpiSettings);
  commandMode() ;
  tftSelect();
  SPI.transfer(0x36);
  dataMode( ) ;
  SPI.transfer(data);
  tftDeselect();
  SPI.endTransaction();  
}

/**
 * @brief 画面の指定範囲を単一色で塗りつぶす
 *
 * RGB565 形式の 16 ビットカラーで画面全ピクセルを書き換えます。
 * 実装は各ピクセルごとに上位・下位バイトを個別に送信します。
 *
 * @param x,y 開始位置
 * @param w,h 塗り潰し範囲 幅、高さ
 * @param Color RGB565 形式の 16 ビット色値
 */
void HX8357::fillBox(uint16_t x , uint16_t y , uint16_t w , uint16_t h , uint16_t Color) {
  uint32_t totalPixels = (uint32_t)w * h;
  const int BUF_PIX = 480;           // 適度なバッファサイズ
  static uint16_t buf[BUF_PIX];
  for (int i = 0; i < BUF_PIX; i++) {
    buf[i] = Color;
  }

  uint16_t xe = x + w -1 ;
  uint16_t ye = y + h -1 ;

  SPI.beginTransaction(hxSpiSettings);
  commandMode() ;
  tftSelect();
  SPI.transfer(0x2A);
  dataMode( ) ;
  SPI.transfer16(x);
  SPI.transfer16(xe);

  commandMode() ;
  SPI.transfer(0x2B);
  dataMode( ) ;
  SPI.transfer16(y);
  SPI.transfer16(ye);

  commandMode() ;
  SPI.transfer(0x2C); // RAMWR
  dataMode( ) ;
  while (totalPixels > 0) {
    uint32_t chunk = (totalPixels > BUF_PIX) ? BUF_PIX : totalPixels;
    SPI.writePixels((uint8_t*)buf, chunk * 2);  // 16bit色 → 2byte
    totalPixels -= chunk;
  }

  tftDeselect();
  SPI.endTransaction();
}

/**
 * @brief 画面全体を単一色で塗りつぶす
 *
 * RGB565 形式の 16 ビットカラーで画面全ピクセルを書き換えます。
 * 実装は各ピクセルごとに上位・下位バイトを個別に送信します。
 *
 * @param Color RGB565 形式の 16 ビット色値
 */
void HX8357::fillScreen(uint16_t Color) {
  fillBox(0,0,_tft_w,_tft_h,Color) ;
}

/**
 * @brief 画面の指定範囲にビットマップ画像を表示する
 *
 * RGB565 形式の 16 ビットカラーで画面全ピクセルを書き換えます。
 * 実装は各ピクセルごとに上位・下位バイトを個別に送信します。
 *
 * @param x,y 開始位置
 * @param w,h 画像サイズ 幅、高さ
 * @param bitmap 画像データ 16 ビット色値
 */
void HX8357::drowBitmap(uint16_t x , uint16_t y , uint16_t w , uint16_t h , uint16_t *bitmap) {
  setWindow(x,y,w,h) ;
  uint16_t xe = x + w -1 ;
  uint16_t ye = y + h -1 ;

  SPI.beginTransaction(hxSpiSettings);
  commandMode() ;
  tftSelect();
  SPI.transfer(0x2A);
  dataMode( ) ;
  SPI.transfer16(x);
  SPI.transfer16(xe);

  commandMode() ;
  SPI.transfer(0x2B);
  dataMode( ) ;
  SPI.transfer16(y);
  SPI.transfer16(ye);

  commandMode() ;
  SPI.transfer(0x2c);

  int len = w*h ;
  dataMode( ) ;
  tftSelect();

  SPI.writePixels((uint8_t*)bitmap, len * 2);  // 16bit色 → 2byte

  tftDeselect();
  SPI.endTransaction();
}

/**
 * @brief 画面の指定範囲にXビットマップ画像を表示する
 *
 * 1ビットモノクロXビットマップデータを受け取り、指定の前景色・背景色で
 * 16ビットカラーに変換して表示します。
 *
 * @param x,y 開始位置
 * @param w,h 画像サイズ 幅、高さ
 * @param bitmap 画像データ 1ビットXビットマップ
 * @param fColor 前景色 RGB565 16ビット色値
 * @param bColor 背景色 RGB565 16ビット色値
 * @param doubleSize true=2倍サイズ表示 false=等倍表示
 */
void HX8357::drawXBitmap(uint16_t x, uint16_t y,uint16_t w, uint16_t h,const uint8_t *bitmap,
                         uint16_t fColor, uint16_t bColor,
                         bool doubleSize) {
  uint16_t xe = x + (doubleSize ? w * 2 : w) - 1;
  uint16_t ye = y + (doubleSize ? h * 2 : h) - 1;
  uint16_t bytesPerLine = (w + 7) >> 3;

  initXbmLUT(fColor, bColor);

  SPI.beginTransaction(hxSpiSettings);

  // --- アドレス指定 ---
  commandMode();
  tftSelect();
  SPI.transfer(0x2A);
  dataMode();
  SPI.transfer16(x);
  SPI.transfer16(xe);

  commandMode();
  SPI.transfer(0x2B);
  dataMode();
  SPI.transfer16(y);
  SPI.transfer16(ye);

  commandMode();
  SPI.transfer(0x2C);   // RAMWR
  dataMode();

  // --- 描画 ---
  for (uint16_t yy = 0; yy < h; yy++) {
    uint16_t outPos = 0;
    const uint8_t *src = bitmap + yy * bytesPerLine;

    // 横方向展開
    for (uint16_t i = 0; i < bytesPerLine; i++) {
      const uint16_t *p = xbmLUT[*src++];
      uint16_t maxOutPos = doubleSize ? w * 2 : w;
      for (int b = 0; b < 8 && outPos < maxOutPos; b++) {
        if (doubleSize) {
          lineBuf[outPos++] = p[b];
          lineBuf[outPos++] = p[b];
        } else {
          lineBuf[outPos++] = p[b];
        }
      }
    }

    uint32_t px = doubleSize ? w * 2 : w;

    // DMA送信（縦2倍は同じ行を2回）
    SPI.writePixels((uint8_t*)lineBuf, px * 2);
    if (doubleSize) {
      SPI.writePixels((uint8_t*)lineBuf, px * 2);
    }
  }

  tftDeselect();
  SPI.endTransaction();
}

/**
 * @brief Xビットマップ用ルックアップテーブル初期化
 * 
 * 1バイト(8ビット)のXビットマップデータを、指定の前景色・背景色で
 * 16ビットカラーに変換するルックアップテーブルを初期化します。
 * 
 * @param fColor 前景色 RGB565 16ビット色値
 * @param bColor 背景色 RGB565 16ビット色値
 */
void HX8357::initXbmLUT(uint16_t fColor, uint16_t bColor) {
  if (colorFG == fColor && colorBG == bColor) { return ; }

  colorFG = fColor ;
  colorBG = bColor ;

  for (int v = 0; v < 256; v++) {
    uint8_t mask = 0x80;
    for (int i = 0; i < 8; i++) {
      xbmLUT[v][i] = (v & mask) ? fColor : bColor;
      mask >>= 1;
    }
  }
}

/**
 * @brief 24ビットRGBカラーを16ビットRGB565カラーに変換する
 *
 * @param r 赤成分 (0-255)
 * @param g 緑成分 (0-255)
 * @param b 青成分 (0-255)
 * @return uint16_t RGB565形式の16ビット色値
 */
uint16_t HX8357::rgb24to565(uint8_t r, uint8_t g, uint8_t b) {
  return ((r & 0xF8) << 8) | ((g & 0xFC) << 3) | (b >> 3);
}

/**
 * @brief BMPファイルをSDカードから読み込み表示する
 * 
 * SDカード上のBMPファイルを読み込み、画面に表示します。
 * 現在は24ビット・16ビットBMPのみ対応しています。
 * BMPファイルは必ず下→上に並んでいる点に注意してください。
 * 
 * @param x,y 表示開始位置
 * @param fileName BMPファイル名（SDカード上のパス）
 */
void HX8357::readBMP(uint16_t x,uint16_t y,char *fileName) {
  // --- SDカード初期化 ---
  fillScreen(COLOR_WHITE);

  File bmp = SD.open(fileName, "r");
  if (!bmp) {
    DPRINTLN("File open failed");
    return;
  }

  // 不要ヘッダ読み飛ばし
  bmp.seek(10);
  uint32_t dataOffset = bmp.read() |
                        (bmp.read() << 8) |
                        (bmp.read() << 16) |
                        (bmp.read() << 24);

  bmp.seek(18);
  int32_t width  = bmp.read() |
                   (bmp.read() << 8) |
                   (bmp.read() << 16) |
                   (bmp.read() << 24);

  int32_t height = bmp.read() |
                   (bmp.read() << 8) |
                   (bmp.read() << 16) |
                   (bmp.read() << 24);

  bmp.seek(28);
  uint16_t bitDepth = bmp.read() | (bmp.read() << 8);

  DPRINTF("BMP: %d x %d  depth=%d  dataOffset=%lu\n",
                width, height, bitDepth, dataOffset);

  if (!(bitDepth == 24 || bitDepth == 16)) {
    DPRINTLN("Unsupported BMP format");
    bmp.close();
    return;
  }

  // --- 表示Wibndow設定-----
  int xe = x+width ;
  int ye = y+height ;
  if (xe > 320) xe = 320 ;
  if (ye > 480) ye = 480 ;

  // ピクセルデータの先頭へ
  bmp.seek(dataOffset);

  // BMPは必ず下→上に並ぶので注意（必要なら上下入れ替えて使用）
  int rowSize = ((bitDepth * width + 31) / 32) * 4; // パディング込み

  uint8_t* rowBuf = (uint8_t*)malloc(rowSize);
  if (!rowBuf) {
    DPRINTLN("Malloc failed");
    bmp.close();
    return;
  }

  uint16_t dispBuff[480] ;
  for (int y = 0; y < height; y++) {
    // 一行読み込み
    bmp.read(rowBuf, rowSize);

    int idx = 0;

    for (int x = 0; x < width; x++) {
      uint16_t pixel565;

      if (bitDepth == 24) {
        // BMP24bitはB,G,Rの順
        uint8_t b = rowBuf[idx++];
        uint8_t g = rowBuf[idx++];
        uint8_t r = rowBuf[idx++];
        pixel565 = rgb24to565(r, g, b);
      } else {
        // 16bit(RGB565) はそのまま
        uint8_t lo = rowBuf[idx++];
        uint8_t hi = rowBuf[idx++];
        pixel565 = (hi << 8) | lo;
      }

      // ---- 出力例：Serialに表示 ----
      dispBuff[x] = pixel565 ;
    }

    drowBitmap(x,ye-y,width,1,dispBuff) ;
  }

  free(rowBuf);
  bmp.close();
  DPRINTLN("Done.");  
}
