/**
 * TFT HX8357 Driver
 * 
 */
#ifndef _HX8357_
#define _HX8357_

#include <arduino.h>
#include <SPI.h>

#define TFT_CS 15
#define TFT_DC 33
#define PIN_CLK 5
#define PIN_MISO 19
#define PIN_MOSI 18

// カラー定義 RGB565形式
#define COLOR_WHITE   0xFFFF
#define COLOR_GRAY    0x7BEF
#define COLOR_DARK_GRAY    0x1863
#define COLOR_BLACK   0x0000  
#define COLOR_RED     0xF800
#define COLOR_GREEN   0x07E0
#define COLOR_DARK_GREEN 0x03E0
#define COLOR_BLUE    0x001F
#define COLOR_YELLOW  0xFFE0
#define COLOR_CYAN    0x07FF
#define COLOR_MAGENTA 0xF81F

// TFT画面サイズ
#define TFT_WIDTH 320
#define TFT_HIGHT 480

// SPI 設定（24MHz を例に。ディスプレイや配線で安定性により下げる）
static const SPISettings hxSpiSettings(25000000, MSBFIRST, SPI_MODE0);

class HX8357 {
  public:
    /** @brief TFT コントローラの初期化を行う */
    void initTFT(int csPin = TFT_CS , int dcPin = TFT_DC) ;

    /** 画面の指定範囲を単一色で塗りつぶす */
    void fillBox(uint16_t x , uint16_t y , uint16_t w , uint16_t h , uint16_t Color) ;

    /** 画面全体を単一色で塗りつぶす */
    void fillScreen(uint16_t Color) ;

    /** 表示方向を設定する */
    void setDir(int dir) ;

    /** @brief 画面の指定範囲にビットマップ画像を表示する */
    void drowBitmap(uint16_t x , uint16_t y , uint16_t w , uint16_t h , uint16_t *bitmap) ;

    /** 画面の指定範囲に2値ビットマップ画像を表示する */
    void initXbmLUT(uint16_t fColor, uint16_t bColor) ;
    void drawXBitmap(uint16_t x, uint16_t y,uint16_t w, uint16_t h,const uint8_t *bitmap,
          uint16_t fColor , uint16_t bColor=COLOR_BLACK , bool doubleSize=false) ;

  private:
    uint16_t colorFG ;  // initXbmLUTで指定した前景色
    uint16_t colorBG ;  // initXbmLUTで指定した背景色
    int _cs ;
    int _dc ;
    int _tft_w ;
    int _tft_h ;

    // 1byte → 8pixel(RGB565)
    uint16_t xbmLUT[256][8];
    uint16_t lineBuf[256];      // 描画用ラインバファ 最大横幅 *2 まで（必要に応じ調整）
    uint16_t dispBitmap[1024] ; // 取り合えず東雲フォントを表示する事しか考えないので最大サイズを16x16Dotとする

    /** 単一のコマンドバイトを送信する */
    void writeCommand(uint8_t cmd) ;

    /** 単一のデータバイトを送信する */
    void writeData(uint8_t data) ;
    void writeData16(uint16_t data) ;

    /** データバッファを連続送信する */
    void writeDataBuffer(const uint8_t *buf, size_t len) ;
    void writeDataBuffer16(const uint16_t *buf, size_t len) ;

    /** Pixcel描画範囲を設定する */
    void setWindow(uint16_t x , uint16_t y , uint16_t w , uint16_t h);

    void commandMode()  { digitalWrite(_dc, LOW); }
    void dataMode()  { digitalWrite(_dc, HIGH); }
    void tftSelect()   { digitalWrite(_cs, LOW); }
    void tftDeselect() { digitalWrite(_cs, HIGH); }

    //============================================================
    //========== SD CARD =========================================
    //============================================================
  public:
    /** SDカードからBMFファイルを読み込んで表示 */
    void readBMP(uint16_t x,uint16_t y,char *fileName) ;

  private:
    /** 24bit → RGB565 変換 */
    uint16_t rgb24to565(uint8_t r, uint8_t g, uint8_t b) ;
};

extern HX8357 tft ;

#endif
