/** お天気アイコン表示関数群
 * 
 */

#include <SD.h>
#include <rom/tjpgd.h>
#include <FS.h>

#include "ImageData.h"
#include "ImageDisp.h"
#include "HX8357.h"

#include "debug.h"

// ▲▼ お天気アイコン ▲▼▲▼▲▼▲▼
static char folderName[] = "/ICON3/" ;
static int bgColor = 0x00 ;
IMAGEDATA imageData ;

/**
 * @brief お天気アイコン表示
 * @param watherCode 気象コード
 * @param x 表示X座標
 * @param y 表示Y座標
 * @param flagNight 夜間フラグ（true:夜間アイコン表示）
 * @return true:成功 false:失敗
 * 
 */
bool LoadWatherIcon(int watherCode, int x, int y, bool flagNight)
{
  DPRINTF("1.Free Heap Size=%d\r\n", esp_get_free_heap_size());
  int frameType = 0 ;
  int code = 0 ;
  int code1 = 0 ;
  int code2 = 0 ;
  int code3 = 0 ;
  for (int i=0;watherCode2Icon[i].watherCode!=0;i++) {
    if (watherCode == watherCode2Icon[i].watherCode) {
      for (int j=0;watherIcon[j].iconNo!=0;j++) {
        if (watherCode2Icon[i].iconDay == watherIcon[j].iconNo) { 
            code = watherIcon[j].iconNo ;
            code1 = watherIcon[j].icon[0] ;
            code3 = watherIcon[j].icon[1] ;
            code2 = watherIcon[j].icon[2] ;
            if (code1 == 1 && flagNight) {
              code1 =5 ;
            }
            if (code3 == 1 && flagNight) {
              code3 =5 ;
            }
            break ;
        }
      }
      if (code != 0) {
        break ;
      }
    }
  }

  char fileName[48];
  if (code2 == 0 && code3 == 0) {
    sprintf(fileName,"%s%d00.jpg",folderName,code1) ;
  } else {
    if (code2 != 0) {
      sprintf(fileName,"%s%dx0.jpg",folderName,code1) ;
      frameType = 1 ;
    } else {
      sprintf(fileName,"%s%d0x.jpg",folderName,code1) ;
      frameType = 2 ;
    }
  }
  if (!LoadImage(fileName,&imageData)) {
      DPRINTF("%s FAIL",fileName) ;
      return false ;
  }

  // --- BASE IMAGE
  uint16_t *baseImage = (uint16_t *)malloc(imageData.getWidth()*imageData.getHight()*2) ;
  memset(baseImage,bgColor,imageData.getWidth()*imageData.getHight()*2) ;
  
  margeIcon(baseImage,(uint16_t *)imageData.getImage(),imageData.getWidth(), imageData.getHight());

  // ---
  if (code2 != 0 || code3 != 0) {
    if (code2 !=0) {
        sprintf(fileName,"%sx%d0.jpg",folderName,code3) ;
    } else {
        sprintf(fileName,"%sxx%d.jpg",folderName,code3) ;
    }
    if (!LoadImage(fileName,&imageData)) {
      DPRINTF("%s FAIL",fileName) ;
        // 2nd image load failed: free base image and ensure any image buffer is released
        imageData.setImage(NULL);
        free(baseImage) ;
        return false ;
    }
    margeIcon(baseImage,(uint16_t *)imageData.getImage(),imageData.getWidth(), imageData.getHight());
  }

  // --------------------------------------------------------------------------------------------
  tft.drowBitmap(x , y ,imageData.getWidth(), imageData.getHight(),baseImage) ;

  // --------------------------------------------------------------------------------------------
  imageData.setImage(NULL) ;
  free(baseImage) ;

  DPRINTF("2.Free Heap Size=%d\r\n", esp_get_free_heap_size());   
  return true ;
}

/**
 * @brief アイコン画像合成
 * @param baseIcon ベースアイコン画像ポインタ
 * @param watherIcon 天気アイコン画像ポインタ
 * @param width 画像幅
 * @param hight 画像高さ
 * 
 */
void margeIcon(uint16_t *baseIcon,uint16_t *watherIcon,int width,int hight) {
  for (int y=0;y<hight;y++) {
      for (int x=0;x<width;x++) {
          uint16_t colorR = (watherIcon[y*width+x] & 0xF100) >> 11 ; // B1111100000011111
          uint16_t colorG = (watherIcon[y*width+x] & 0x07E0) >> 5 ;  // B0000011111100000
          uint16_t colorB = watherIcon[y*width+x] & 0x001F ;  // B0000000000011111
          if (colorR > 16 || colorG > 16 || colorB > 16) {  // 黒ぽい色を透過色として扱う
              baseIcon[y*width+x] = watherIcon[y*width+x] ;
          }
      }
  }
}

/**
 * @brief アイコン画像枠合成
 * @param baseIcon ベースアイコン画像ポインタ
 * @param watherIcon 天気アイコン画像ポインタ
 * @param width 画像幅
 * @param hight 画像高さ
 * @param frameColor 枠色
 * 
 */
void margeFrame(uint16_t *baseIcon,uint16_t *watherIcon,int width,int hight,uint16_t frameColor) {
  for (int y=0;y<hight;y++) {
      for (int x=0;x<width;x++) {
          uint16_t colorR = (watherIcon[y*width+x] & 0xF100) >> 11 ; // B1111100000011111
          uint16_t colorG = (watherIcon[y*width+x] & 0x07E0) >> 5 ;  // B0000011111100000
          uint16_t colorB = watherIcon[y*width+x] & 0x001F ;  // B0000000000011111
          if (colorR > 16 || colorG > 16 || colorB > 16) {  // 黒ぽい色を透過色として扱う
              baseIcon[y*width+x] = frameColor ;
          }
      }
  }
}

// △▽ SD CARD △▽△▽△▽△▽

/** ユーザ定義デバイス構造体 */
typedef struct {
  const void *fp; /* File pointer for input function */
  BYTE *fbuf;     /* Pointer to the frame buffer for output function */
  UINT wfbuf;     /* Width of the frame buffer [pix] */
} IODEV;

// △▽ JPEG LOAD △▽△▽△▽△▽
/** JPEG入力関数
 *
 * TJpgDec ライブラリから呼び出され、JPEGデータを読み込みます。
 *
 * @param jd TJpgDec デコーダオブジェクトへのポインタ
 * @param buff データ読み込み先バッファへのポインタ（NULL の場合はデータ破棄）
 * @param nbyte 読み込むバイト数
 * @return 実際に読み込んだバイト数
 */
UINT in_func(JDEC* jd, BYTE* buff, UINT nbyte)
{
  UINT ret = 0;
  IODEV *dev = (IODEV*)jd->device;
  File *f2 = (File *)dev->fp;
  if (buff) {
    UINT R = (UINT)f2->read(buff, nbyte);
    return R ;
  } else {
    // Remove bytes from input stream 
    if(f2->seek(nbyte, SeekCur)){ //seek mode:SeekCur=1(FS.h)
      ret = nbyte;
    }else{
      ret = 0;
    }
  }
  return ret;
}
 
/** JPEG出力関数
 *
 * TJpgDec ライブラリから呼び出され、デコードした画像データをフレームバッファに出力します。
 *
 * @param jd TJpgDec デコーダオブジェクトへのポインタ
 * @param bitmap 出力するビットマップデータへのポインタ
 * @param rect 出力する矩形領域情報へのポインタ
 * @return 1:続行 0:中止
 */
UINT out_func (JDEC* jd, void* bitmap, JRECT* rect)
{
  IODEV *dev = (IODEV*)jd->device;
  uint8_t *src, *dst;
  uint16_t x, y, bwd;
 
  src = (uint8_t*)bitmap;
  //use RGB565
  dst = dev->fbuf + 2 * (rect->top * dev->wfbuf + rect->left);
  //bws = 2 * (rect->right - rect->left + 1);
  bwd = 2 * dev->wfbuf;

  uint8_t msb, lsb, tmpR, tmpG , tmpB;
  int16_t xcnt;
  for (y = rect->top; y <= rect->bottom; y++) {
    xcnt = 0;
    for (x = rect->left; x <= rect->right; x++){
      tmpR = *(src++) & 0b11111000;
      tmpG = *(src++) & 0b11111100;
      tmpB = *(src++) & 0b11111000;
      msb = tmpR | (tmpG >> 5);
      lsb = (tmpG << 3) | (tmpB >> 3);
      *(dst + xcnt++) = lsb;
      *(dst + xcnt++) = msb;
    }
    dst += bwd;
  }
 
  return 1;    // Continue to decompress 
}

/**
 * @brief JPEG画像読み込み
 * @param fileName JPEGファイル名
 * @param imageData 画像データ格納先ポインタ
 * @return true:成功 false:失敗
 * 
 */
bool LoadImage(char *fileName,IMAGEDATA *imageData) {
  void *work;       /* Pointer to the decompressor work area */
  JDEC jdec;        /* Decompression object */
  JRESULT res;      /* Result code of TJpgDec API */
  IODEV devid;      /* User defined device identifier */

  bool result = true ;

  init( ) ;
  File file = SD.open(fileName);
  if(!file){
    DPRINTLN("Failed to open file for reading");
    return false;
  }
  // Use DPRINTF for debug prints; controlled by DEBUG macro in debug.h
  DPRINTF("%s file size = %d\r\n", fileName, file.size());
  DPRINTF("1.Free Heap Size=%d\r\n", esp_get_free_heap_size());

  devid.fp = &file;
 
  // Allocate a work area for TJpgDec 
  const int WorkBuffSize = 1024*4 ;
  work = malloc(WorkBuffSize);
  // Prepare to decompress 
  res = jd_prepare(&jdec, in_func, work, WorkBuffSize, &devid);
  if (res == JDR_OK) {
    // Ready to dcompress. Image info is available here. 
    imageData->setWidth(jdec.width) ;
    imageData->setHight(jdec.height) ;
    imageData->allocImage(2 * jdec.width * jdec.height) ; //RGB565
    devid.fbuf = (BYTE *)imageData->getImage( ) ;
    if (devid.fbuf != NULL) {
      devid.wfbuf = jdec.width;
  
      res = jd_decomp(&jdec, out_func, 0);   // Start to decompress with 1/1 scaling 
      if (res == JDR_OK) {
        // Decompression succeeded. You have the decompressed image in the frame buffer here. 
        DPRINTF("\rOK  \n");
      } else {
        DPRINTF("Failed to decompress: rc=%d\n", res);
      }
    } else {
      DPRINTF("Out of Memory\r\n");
      result = false ;
    }
  } else {
    DPRINTF("Failed to prepare: rc=%d\n", res);
    result = false ;
  }
  free(work);             // Discard work area 
  work = NULL ;

  if (!result) {
    // If decompression failed, ensure any partially allocated image buffer is freed
    imageData->setImage(NULL);
    file.close();
    return result ;
  }
 
  DPRINTF("2.Free Heap Size=%d\r\n", esp_get_free_heap_size());
  if(devid.fbuf){
    devid.fbuf = NULL;
    DPRINTLN("Free devid.fbuf OK!");
  }else{
    DPRINTLN("Failed free devid.fbuf.");
  }

  DPRINTF("3.Free Heap Size=%d\r\n", esp_get_free_heap_size());
  return result ;
}
