// 気象庁　天気予報データ取得
#include <WiFi.h>
#include <HTTPClient.h>
#include <ArduinoJson.h>
#include "JMA.h"
#include "property.h"
#include "debug.h"

JMAWATHER jmaWather ;

/**
 * @brief WiFi接続を行う
 * @return true:接続成功 false:接続失敗
 * 
 */
bool JMAWATHER::connectWiFi( ) {
  DPRINTLN("connectWiFi") ;
  if (WiFi.status() != WL_CONNECTED) {
    // --- Netwaorkに接続
    DPRINTLN("Networkに接続") ;
    WiFi.begin(property.getWifiAP( ) , property.getWifiPWD( ));

    int i = 0 ;
    while (WiFi.status() != WL_CONNECTED) {
      delay(500);
      i++ ;
      if (i > 20) {
        delay(30000);
        DPRINTLN("connectWiFi - false") ;
        return false ;
      }
    }
    DPRINTLN("");
    DPRINTLN("WiFi connected");
    DPRINT("IP address: ");
    DPRINTLN(WiFi.localIP());
  }
  return true ;
}

/**
 * @brief NTPサーバーから時刻取得
 * @return true:成功 false:失敗
 * 
 */
bool JMAWATHER::getNTP( ) {
  if (!connectWiFi()) {
    return false;
  }
  configTime(9 * 3600L, 0, "ntp.nict.jp", "time.google.com", "ntp.jst.mfeed.ad.jp");
  getLocalTime(&ntpTime) ;

  return true;
}

/**
 * @brief 日付文字列をWatherTime_t形式に変換
 * @param dateStr 日付文字列
 * @return WatherTime_t形式の日付データ
 * 
 */
void JMAWATHER::initWatherData( ) {
    for (int i=0;i<2;i++) {
    watheData[i].watherCode = 0 ;
    memset(watheData[i].wather,0,sizeof(watheData[i].wather)) ;
    for (int j=0;j<4;j++) {
      watheData[i].pops[j] = -1 ;
    }
  }
}

/**
 * @brief 天気予報データ取得
 * @return true:成功 false:失敗
 * 
 */
bool JMAWATHER::getWatherData( ) {
  if (!connectWiFi()) {
    return false ;
  }
  HTTPClient http;
  String jmaUrl = property.getWeatherURL( ) + property.getRegionCode( ) + ".json" ;
  DPRINTLN(jmaUrl) ;
  
  http.begin(jmaUrl);         // URLを指定
  int httpCode = http.GET();  // GETリクエストを送信
  if (httpCode == 200) {      // Request Result 正常終了
      readWatherData(http.getString(),property.getAreaCode( )) ;
  } else {
    DPRINT("http Rewquest failed. -- ");
    DPRINTLN(httpCode);
    return false ;
  }
  return true ;
}

/**
 * @brief 天気予報データ解析
 * @param watherDataStr 天気予報JSONデータ文字列
 * @param areaCode 予報区コード
 * 
 */
void JMAWATHER::readWatherData(String watherDataStr,int areaCode) {
  memset(watheData,0,sizeof(watheData)) ;
  DPRINTLN("--- JSON DECODE ---");
  JsonDocument doc;
  DeserializationError error = deserializeJson(doc, watherDataStr);
  DPRINTF("deserializeJson=%d\n",error);

  // ----- JSON DUMP
  DPRINTLN("--- JSON DUMP ---");
  bool flagDailyWatherData = false ;
  initWatherData( ) ;

  char *strCharAddr ;
  for (int pubOff = 0 ; flagDailyWatherData == false ; pubOff ++) {
    DPRINTF("=== pubOff:%d\n",pubOff) ;
    // -- 観測エリア
    if (doc[pubOff]["publishingOffice"].as<String>()== "null") { break ; }
    copyJString(publishingOffice,doc[pubOff]["publishingOffice"].as<const char*>()) ;
    DPRINTF("=== pubOff:%d\n",pubOff) ;
    DPRINTF("publishingOffice=%s\n",publishingOffice);
    DPRINTF("reportDatetime=%s\n",doc[pubOff]["reportDatetime"].as<const char*>());
    // -- 予報日時 reportDatetime
    WatherTime_t reportDatetime = dateFormat(doc[pubOff]["reportDatetime"].as<const char*>()) ;

    for (int timSer = 0 ; ; timSer ++) {
      if (doc[pubOff]["timeSeries"][timSer].as<String>()== "null") { break ; }

      // --- 天気予報情報判定
      bool flagWatherData = false ;
      bool flagPopsData = false ;
      bool flagTempsData = false ;
      if (doc[pubOff]["timeSeries"][timSer]["areas"][0]["weathers"].as<String>() != "null") {
        flagWatherData = true ;
        flagDailyWatherData = true ;
      } else
      if (doc[pubOff]["timeSeries"][timSer]["areas"][0]["pops"].as<String>() != "null") {
        flagPopsData = true ;
      } else
      if (doc[pubOff]["timeSeries"][timSer]["areas"][0]["temps"].as<String>() != "null") {
        flagTempsData = true ;
      } else {
        break ;
      }

      int timdefNum = 0 ;
      for (int area = 0 ; ; area++) {
        if (doc[pubOff]["timeSeries"][timSer]["areas"][area].as<String>() == "null") break ;
        if (doc[pubOff]["timeSeries"][0]["areas"][area]["area"]["code"].as<int>() == areaCode) {
          for (int timdef = 0 ; ; timdef ++) {
              if (doc[pubOff]["timeSeries"][timSer]["timeDefines"][timdef].as<String>() == "null") break ;
              timdefNum ++ ;
              WatherTime_t dtm = dateFormat(doc[pubOff]["timeSeries"][timSer]["timeDefines"][timdef].as<const char*>()) ;
              if (flagWatherData && timdef < 2) {
                // --- 天気予報情報 : 今日・明日 ---
                if (doc[pubOff]["timeSeries"][timSer]["areas"][area]["weathers"][timdef].as<String>() == "null") break ;
                if (doc[pubOff]["timeSeries"][timSer]["areas"][area]["weatherCodes"][timdef].as<String>() == "null") break ;
                    // -- 予報区
                    copyJString(areaName,
                                doc[pubOff]["timeSeries"][timSer]["areas"][area]["area"]["name"].as<const char*>());
                    watheData[timdef].watherTime = dtm ;
                    watheData[timdef].watherCode = doc[pubOff]["timeSeries"][timSer]["areas"][area]["weatherCodes"][timdef].as<int>() ;
                    copyJString(watheData[timdef].wather,
                                doc[pubOff]["timeSeries"][timSer]["areas"][area]["weathers"][timdef].as<const char*>(),WATHE_LEN) ;
              } else
              if (flagPopsData) {
                // --- 降水確率 ---
                if (doc[pubOff]["timeSeries"][timSer]["areas"][area]["pops"][timdef].as<String>() == "null") break ;
                    int day = 0 ;
                    if (watheData[0].watherTime.tm_mday == dtm.tm_mday) day = 0 ;
                    else if (watheData[1].watherTime.tm_mday == dtm.tm_mday) day = 1 ;
                    else break ;
                    int tmPos = dtm.tm_hour / 6 ;  // 0時6時12時18時の6時間区切りになっている事を前提としている
                    watheData[day].pops[tmPos] = doc[pubOff]["timeSeries"][timSer]["areas"][area]["pops"][timdef].as<int>() ;
              } else
              if (flagTempsData) {
                if (doc[pubOff]["timeSeries"][timSer]["areas"][area]["temps"][timdef].as<String>() == "null") break ;
              } else {
                break ;
              }
          } // Next timdef
        } // endif
      } // Next area
    } // Next timSer
  } //  Next pubOff

  DPRINTF("\n--- 天気予報情報 ---\n%s - %s\n",publishingOffice,areaName);
  for (int day=0;day<2;day++) {
      DPRINTF("%02d/%02d %3d\n  %s\n",
        watheData[day].watherTime.tm_mon,watheData[day].watherTime.tm_mday,
        watheData[day].watherCode,watheData[day].wather) ;
      DPRINT("  降水確率:");
      for (int pops = 0 ; pops < 4;pops++) {
        if (watheData[day].pops[pops] < 0) {
          DPRINT(" -- ") ;
        } else {
          DPRINTF(" %2d ",watheData[day].pops[pops]) ;
        }
      }
      DPRINTLN() ;
      DPRINTLN() ;
  }

}

/**
 * @brief 日付文字列をWatherTime_t形式に変換
 * @param dateStr 日付文字列
 * @return WatherTime_t形式の日付データ
 * 
 */
WatherTime_t JMAWATHER::dateFormat(String dateStr) {
  WatherTime_t tim ;
  int dateData[6] ;
  String dateDisp = "" ;
  int l = dateStr.length( ) ;
  int p = 0 ;
  for (int i = 0; i < 6 ; i++) {
    dateData[i] = 0 ;
  }
  for (int j=0;j<l;j++) {
    if (dateStr.charAt(j) == '+') break ;
    if ('0' <= dateStr.charAt(j) && dateStr.charAt(j) <= '9') {
      dateData[p] = dateData[p] * 10 + (dateStr.charAt(j) - '0') ;
    } else {
      p ++ ;
    }
  }
  tim.tm_year = dateData[0] ;  // 年
  tim.tm_mon  = dateData[1];   // 月 - 1 (0-11)
  tim.tm_mday = dateData[2];   // 日 (1-31)
  tim.tm_hour = dateData[3];   // 時 (0-23)
  tim.tm_min  = dateData[4];   // 分 (0-59)
  tim.tm_sec  = dateData[5];   // 秒 (0-59)
  
  return tim ;
}

/**
 * @brief UTF-8文字列のバイト長を取得
 * @param utf8codes UTF-8文字列ポインタ
 * @return 文字長（バイト数）
 *          UTF8の文字長(2 or 3)を返す
 *          ASCIIの場合は1を返す
 * 
 */
int JMAWATHER::UTF8LEN(uint8_t *utf8codes) {
  int utfLen = 0 ;
  if( utf8codes[0] < 0x80 ){
    utfLen = 1 ;
  } else
  if( utf8codes[0] < 0xE0 ){
    utfLen = 2 ;
  } else {
    utfLen = 3 ;
  }
  return utfLen ;
}

/**
 * @brief JString文字列をC文字列にコピー
 * @param dst コピー先C文字列ポインタ
 * @param src コピー元JString文字列ポインタ
 * @param maxLen コピー最大長
 * @return コピー後の文字列長
 * 
 */
int JMAWATHER::copyJString(char *dst , const char *src , int maxLen) {
  memset(dst,0,maxLen) ;
  int pos = 0 ;
  while (src[pos] != 0) {
    int len = UTF8LEN((uint8_t *)&src[pos]) ;
    if (pos + len >= maxLen) { break ; }
    dst[pos] = src[pos] ; pos ++ ;
    if (len >= 2) { dst[pos] = src[pos] ; pos ++ ; }
    if (len == 3) { dst[pos] = src[pos] ; pos ++ ; }
  }
  return pos ;
}
