/**
 * タッチスクリーン読み取り
 */
#include <Arduino.h>
#include <Wire.h>
#include "debug.h"

#define TS_MINX 150
#define TS_MINY 130
#define TS_MAXX 3800
#define TS_MAXY 4000

volatile bool touchDetected = false;  // タッチ検出フラグ

/**
 * @brief タッチ検出状態を取得する
 * 
 * @return true: タッチ検出あり
 * @return false: タッチ検出なし
 */
bool isTouch( ) {
  if (touchDetected){
    touchDetected = false;
    return true ;
  }
  return false ;
}

#define STMPE_CS 32 // TSC2007 IRQ Pin
#define TSC2007_ADDR 0x48 //タッチスクリーンI2Cアドレス

// TSC2007 コマンド定義
#define CMD_X     0xC0   // X位置読み取り
#define CMD_Y     0xD0   // Y位置読み取り
#define CMD_Z1    0xE0   // Z1
#define CMD_Z2    0xF0   // Z2
#define CMD_POWER 0x00   // 電源制御

/**
 * @brief タッチ検出割り込みサービスルーチン
 * 
 */
void IRAM_ATTR penirqISR() {
  touchDetected = true;   // フラグをセット
}

/**
 * @brief TSC2007 から12ビットADCデータを取得する
 * 
 * @param command コマンドコード
 * @return 12ビットADCデータ
 */
uint16_t tscRead(uint8_t command) {
  Wire.beginTransmission(TSC2007_ADDR);
  Wire.write(command);
  Wire.endTransmission();

  Wire.requestFrom(TSC2007_ADDR, 2);
  uint16_t v = 0;
  if (Wire.available() >= 2) {
    v = (Wire.read() << 4) | (Wire.read() >> 4);
  }
  return v;
}

/**
 * @brief タッチ座標を取得する
 * 
 * @param x 取得したX座標格納先
 * @param y 取得したY座標格納先
 * @return true:タッチ検出あり false:タッチ検出なし
 */
bool readTouch(uint16_t &x,uint16_t &y) {
  // タッチされていなければスキップ
  if (digitalRead(STMPE_CS) == HIGH) {
    return false ;
  }
  // X, Y 座標を取得
  int sx = tscRead(CMD_X);
  int sy = tscRead(CMD_Y);
  uint16_t z1 = tscRead(CMD_Z1);
  uint16_t z2 = tscRead(CMD_Z2);

  // タッチ圧から有効タッチを判断 (簡易)
  bool touched = (z1 > 100);

  y = 320-map(sx, TS_MINX, TS_MAXX, 0, 320);
  x = map(sy, TS_MINY, TS_MAXY, 0, 480);

  if (touched) {
    DPRINTF("X = %3d",x); 
    DPRINTF("   Y = %3d",y);
    DPRINTF("   sx = %3d",sx);
    DPRINTF("   sy = %3d\n", sy);
  }
  return touched ;
}

/**
 * @brief TSC2007 タッチスクリーンの初期化
 * 
 */
void init_TSC( ) {
  Wire.begin();
  DPRINTLN("TSC2007 Touch Test");

  // LOW エッジ割り込み（タッチした瞬間）
  pinMode(STMPE_CS, INPUT_PULLUP);
  attachInterrupt(digitalPinToInterrupt(STMPE_CS), penirqISR, FALLING);
}
