/* TWELITE PAL 環境センサー通信 */

#include <Arduino.h>
#include "Twelite.h"
#include "debug.h"

TWELITE twelite ;

TWELITE::TWELITE( ) {
  recvLen = 0 ;  // 受信したデータのバイト数
  recvFlag = false ;
  recvMSB = true ;
}

/** TWELITE シリアル通信初期化 */
void TWELITE::init( ) {
  Serial1.begin(115200, SERIAL_8N1, TWELITE_RX, TWELITE_TX);  // シリアル通信1初期化(RX, TX)
}

/**
 * １Byteのデータを16進数2文字のASCII文字列に変換。
 * ターミネーター(NULL)を付与する。
 */
void TWELITE::hexCharaSet(uint8_t data,char *buffPos) {
  static char bin2Hex[] = {"0123456789ABCDEF"} ;

  buffPos[0] = bin2Hex[data >> 4] ;
  buffPos[1] = bin2Hex[data & 0x0F] ;
  buffPos[2] = 0x00 ;
}

/**
 * 16進数1桁のASCII文字を数値に変換。
 */
uint8_t TWELITE::hex2Bin(char c) {
  uint8_t hex ;
  if ('0' <= c && c <= '9') {
    hex = c - '0' ;
  } else
  if ('A' <= c && c <= 'F') {
    hex = c - 'A' + 10 ;
  } else
  if ('a' <= c && c <= 'f') {
    hex = c - 'a' + 10 ;
  }
  return hex ;
}

/**
 * TWELITE PAL 環境センサーからの受信データをデコードして
 * twelitePalEnv に格納する。
 * return : -1: - チェックサムエラー 0> envNo(PAL1 / PAL2)
 */
int TWELITE::readDataDecode(int dataLen , uint8_t *sensorData) {
  int envNo = - 1;

  #ifdef DEBUG
  char dispStr[4] ;
  Serial.print(":") ;
  for (int i=0;i<dataLen;i++) {
    sprintf(dispStr,"%02X",sensorData[i]) ;
    Serial.print(dispStr) ;
  }
  Serial.println() ;
#endif

  // -- CheckSum
  uint8_t ch1 = 0 ; // チェックサム１：CRC8
  uint8_t ch2 = 0 ; // チェックサム２：LRC
  uint8_t MSB_CRC8 = 0x31; // X^8+X^5+X^4+1 : CRC8-Maxim
  for (int i=0;i<dataLen-2;i++) {
    ch1 ^= sensorData[i] ;
    for ( int b = 0 ; b < CHAR_BIT ; b++ ){
      if ( ch1 & 0x80 ){
        ch1 = (ch1 << 1) ^ MSB_CRC8;
      } else {
        ch1 <<= 1;
      }
    }
  }
  for (int i=0;i<dataLen-1;i++) {
    ch2 += sensorData[i] ;
  }
  ch2 = ~ch2 + 1 ;
  if (ch1 != sensorData[dataLen-2] || ch2 != sensorData[dataLen-1]) {
    DPRINTLN("CHECK SUM ERROR") ;
    return envNo;
  }

  // 送信デバイスID
  int devId = sensorData[11] ;
  envNo = (devId == 1) ? PAL1 : PAL2 ;
  memset((char *)&twelitePalEnv[envNo],0,sizeof(struct TwelitePalEnv_s)) ;
  twelitePalEnv[envNo].devId = sensorData[11] ;
  // 送信元個体識別番号
  for (int i=0;i<4;i++) {
    twelitePalEnv[envNo].serialNo[i] = sensorData[i+7] ;
  }
  // 電波強度
  twelitePalEnv[envNo].lqi = sensorData[4] ;

  // --- チャンク
  int cDnum = sensorData[14] ;
  int dp = 15 ;
  for (int cn = 0 ; cn < cDnum ; cn ++) {
    int16_t data16 = 0 ;
    uint16_t dataU16 = 0;
    int32_t data32 = 0 ;

    uint8_t dataType = sensorData[dp] ; dp++ ;
    uint8_t dataSource = sensorData[dp] ; dp++ ;
    uint8_t exData = sensorData[dp] ; dp++ ;
    uint8_t dataLen = sensorData[dp] ; dp++ ;

    DPRINT("cn=") ;
    DPRINT(cn) ;
    DPRINT("  DataBit=") ;
    DPRINTLN(readData[dp],HEX) ;
    DPRINT("dataType=") ;
    DPRINT(dataType,HEX) ;
    DPRINTLN("  dataLen=") ;
    DPRINT(dataLen,HEX) ;
    DPRINT("  dataSource=") ;
    DPRINT(dataSource,HEX) ;
    DPRINT("  exData=") ;
    DPRINTLN(exData,HEX) ;

    if (dataLen == 2) {
      ((int8_t *)&data16)[1] = sensorData[dp] ; ((uint8_t *)&dataU16)[1] = sensorData[dp] ; dp++ ;
      ((int8_t *)&data16)[0] = sensorData[dp] ; ((uint8_t *)&dataU16)[0] = sensorData[dp] ; dp++ ;
    } else
    if (dataLen == 4) {
      ((uint8_t *)&data32)[3] = sensorData[dp] ; dp++ ;
      ((uint8_t *)&data32)[2] = sensorData[dp] ; dp++ ;
      ((uint8_t *)&data32)[1] = sensorData[dp] ; dp++ ;
      ((uint8_t *)&data32)[0] = sensorData[dp] ; dp++ ;
    } else {
      dp += dataLen ;
    }
    DPRINTLN("data16=") ;
    DPRINT(data16) ;
    DPRINT("  dataU16=") ;
    DPRINT(dataU16) ;
    DPRINT("  data32=") ;
    DPRINTLN(data32) ;

    if (dataSource == 0x30 && exData == 0x08) {
      // 電源電圧
      twelitePalEnv[envNo].volt = dataU16 ;
    } else
    if (dataSource == 0x01 && exData == 0x00) {
      // 温度
      twelitePalEnv[envNo].temperature = data16 ;
    } else
    if (dataSource == 0x02 && exData == 0x00) {
      // 湿度
      twelitePalEnv[envNo].humidity = dataU16 ;
    } else
    if (dataSource == 0x03 && exData == 0x00) {
      // 明るさ
      twelitePalEnv[envNo].brightness = data32 ;
    }
  }
  DPRINTLN("  temperature=") ;
  DPRINT(twelitePalEnv[envNo].temperature) ;
  DPRINT("  humidity=") ;
  DPRINT(twelitePalEnv[envNo].humidity) ;
  DPRINT("  brightness=") ;
  DPRINTLN(twelitePalEnv[envNo].brightness) ;

  return envNo ;
}

/**
 * TWELITE シリアルデータ受信処理
 * return : -1: 受信なし 0>: envNo(PAL1 / PAL2)
 */
int TWELITE::readSerial( ) {
  int envNo = -1 ;
  // データ受信
  if (Serial1.available() > 0) {
    char c = Serial1.read();
    if (c == ':') {
      memset(readData,0,READ_DATA_LEN) ;
      recvFlag = true ;
      recvLen = 0 ;
    } else
    if (c == 0x0d) {
      // STATUS RECIVE
      recvFlag = false ;
      if (recvLen == 49 && readData[12] == 0x80 && readData[13] == 0x82) {
        // 環境センサーPAL
        envNo = readDataDecode(recvLen , readData) ;
      }
    } else {
      if (c >= ' ') {
        if (recvMSB) {
          readData[recvLen] = hex2Bin(c) << 4 ;
          recvMSB = false ;
        } else {
          readData[recvLen] |= hex2Bin(c) ;
          recvMSB = true ;
          recvLen++ ;
        }
      }
    }
  }
  return envNo ;
}

// 送信デバイスID
int TWELITE::getDevId(int PalNo) {
  return twelitePalEnv[PalNo].devId ;
}

// 電波強度
int TWELITE::getLqi(int PalNo) {
  return twelitePalEnv[PalNo].lqi ;  
}

// 電源電圧
float TWELITE::getVolt(int PalNo) {
  return twelitePalEnv[PalNo].volt / 1000. ;  
}

// 明るさ
int TWELITE::getBrightness(int PalNo) {
  return twelitePalEnv[PalNo].brightness ;  
}

// 温度
float TWELITE::getTemperature(int PalNo) {
  return twelitePalEnv[PalNo].temperature /100.;
}

// 湿度
float TWELITE::getHumidity(int PalNo) {
  return twelitePalEnv[PalNo].humidity /100.;
}
