/**
 * 天気予報 ＆ 気温表示
 * HX8357 & Adafruit ESP32 Feather
 * BME280 & TwelitePAL
 */
#include <time.h>
#include <WiFi.h>

#include "debug.h"

#include "SDCard.h"
#include "HX8357.h"
#include "FontSD.h"
#include "TSC.h"
#include "property.h"

#include "ImageDisp.h"
#include "DISP7SEG.h"
#include "JMA.h"
#include "BME280.h"
#include "Twelite.h"
#include "AreaCodeSel.h"
#include "ARAM.h"


// --- グローバル状態フラグ ---
bool flagNTP ;       // NTP同期状態フラグ（true: NTPから時刻取得済み）
bool flagWATHER ;    // 天気データ取得フラグ（true: 気象データ取得完了）
int  statWATHER = 0; // 気象データ状態 0:未取得 1:取得済 2:取得待ち

// 気圧グラフ表示用
#define airPressMax 34    // 気圧データ最大保存数
int airPressTop = 0 ;     // 気圧データ保存位置
int airPressNum = 0 ;     // 保存済み気圧データ数
int airPressWide = 4 ;    // 気圧グラフ表示幅
int airPress[airPressMax] ;  // 気圧データ保存配列

// --- 一時表示制御用変数 ---
char str[48] ;      // str: 各種表示用の一時文字列バッファ
int lastSec = 0 ;   // 最後に更新した秒（秒単位の変更検出用）

// --- 選択／表示モード ---
int controlMode = 0 ;  // 制御モード 0:天気予報表示 1:都道府県選択 2:市町村選択
int officeSel = 0 ;    // 選択中の officeCode（地域選択用）
int areaSel = 0 ;      // 選択中の areaCode（地域選択用）
int aramSel = 0 ;      // 選択中のアラーム番号

// ▲▼▲▼▲▼▲▼▲▼▲▼▲▼▲▼▲▼▲▼
TaskHandle_t AramTaskHandle = NULL;
int AramTaskMode = 0 ;
void AramTask(void *pvParameters) {
  while (1) {
    // === 通知が来るまで待機 ===
    ulTaskNotifyTake(pdTRUE, portMAX_DELAY);
    if (1 <= AramTaskMode && AramTaskMode <= 4) {
      aram.playChime(AramTaskMode) ;
    }
    Serial.println("[AramTask] done");
    AramTaskMode = 0 ;
  }
}

// ▲▼▲▼▲▼▲▼▲▼▲▼▲▼▲▼▲▼▲▼
/**
 * @brief 降水確率表示
 * @param x 表示X座標
 * @param y 表示Y座標
 * @param day 0-今日 1-明日
 * 
 */
void dispPops(int x, int y , int day) {
  // text: 各時間帯（朝・昼・夕・夜）の降水確率を文字列化して保持
  char text[4][4] ;
  // dispText: 画面表示用に結合した降水確率文字列バッファ
  char dispText[24] ;
  for (int i=0;i<4;i++) {
    if (jmaWather.watheData[day].pops[i] < 0) {
      sprintf(text[i],"---") ;
    } else {
      sprintf(text[i],"%3d",jmaWather.watheData[day].pops[i]) ;
    }
  }
  sprintf(dispText,"%s %s %s %s",text[0],text[1],text[2],text[3]) ;
  fontSD.dispTextG(x , y , 146 , dispText) ;
}

/**
 * @brief 天気表示
 * 
 */
void dispWather( ) {
  if (controlMode != 0) return ;
  
  // --- Disp Clock ---
  struct tm timeinfo;
  getLocalTime(&timeinfo);
  // timeinfo: 現在のローカル時刻を保持
  disp7seg.dispTime(2, 2 ,timeinfo.tm_year + 1900 , timeinfo.tm_mon+1 , timeinfo.tm_mday , timeinfo.tm_hour , timeinfo.tm_min ,timeinfo.tm_sec , 0xFFFF , 0x0000) ; 
  // text: 地域名／市名を結合して表示するためのバッファ
  char text[80] ;
  sprintf(text,"%s %s",(char *)property.getRegionName().c_str(),(char *)property.getCityName().c_str()) ;
  fontSD.dispTextG(20   , 38 , 320 , text , COLOR_GREEN ) ;
  // flagKnight: 昼夜の判定（true: 夜）
  bool flagKnight = (6 <= timeinfo.tm_hour && timeinfo.tm_hour <= 18) ? false : true ;
  // 天気アイコン＆降水確率表示
  LoadWatherIcon(jmaWather.watheData[0].watherCode, 2  ,  54, flagKnight) ;
  dispPops(20 , 166 , 0) ;
  LoadWatherIcon(jmaWather.watheData[1].watherCode, 2  , 192, flagKnight) ;
  dispPops(20 , 304 , 1) ;
  // 天気概要表示
  tft.fillBox(164,54,144,138,COLOR_BLACK) ;
  fontSD.dispTextG(164 , 54 , 128 , (char *)jmaWather.watheData[0].wather,COLOR_WHITE,COLOR_BLACK,true) ;
  tft.fillBox(164,192,144,138,COLOR_BLACK) ;
  fontSD.dispTextG(164 ,192 , 128 , (char *)jmaWather.watheData[1].wather,COLOR_WHITE,COLOR_BLACK,true) ;  
}

/**
 * @brief BME280 環境値表示
 * @param temp 温度（℃）
 * @param pres 気圧（hPa）
 * @param hum 湿度（%）
 */
void dispBME280(float temp,float pres,float hum) {
  if (controlMode != 0) return ;

  DPRINTF("温度: %f ℃  ",temp);
  DPRINTF("湿度: %f %" ,hum);
  DPRINTF("気圧: %d hPa\n",pres);
  DPRINTLN("--------------------");
  sprintf(str,"%s 気圧%6.1fhPa",property.getPalName(0) , pres) ;
  fontSD.dispTextG(308 , 54 , 172 , str , COLOR_WHITE) ; 
  int x = disp7seg.dispCompensate(308, 74 , 0 , temp , COLOR_WHITE , COLOR_BLACK) ;
      x = disp7seg.dispCompensate(x+16 , 74 , 1 , hum , COLOR_WHITE , COLOR_BLACK) ;
}

// 点滅用アイコンカラー制御変数
int loopCount = 0 ;

/**
 * @brief 点滅用アイコンカラー取得
 * 
 * @return uint16_t アイコンカラー
 */
uint16_t blinkIconColor() {
  uint16_t color = COLOR_RED ;
  loopCount = (loopCount + 1) % 3 ;
  switch(loopCount) {
    case 0: color = COLOR_YELLOW ; break ;
    case 1: color = COLOR_MAGENTA ; break ;
    case 2: color = COLOR_CYAN ; break ;
  }
  return color ;
}

/**
 * @brief TWELITE 環境値表示
 * @param envNo 環境センサー番号
 */
void dispTWELITE(int envNo) {
  if (controlMode != 0) return ;

  DPRINTF("DEV=%d  DevID=%d  temperature=%2.1f   humidity=%2.1f  brightness=%5d\n", 
          envNo , twelite.getDevId(envNo) , twelite.getTemperature(envNo) , twelite.getHumidity(envNo) , twelite.getBrightness(envNo) ) ;
  if (envNo == 0) {
    sprintf(str,"%s 照度%5dLx",property.getPalName(1),twelite.getBrightness(PAL1)) ;
            fontSD.dispTextG(308 , 110 , 320 , str , COLOR_YELLOW) ; 
    // 補正済み環境値表示
    int x = disp7seg.dispCompensate(308, 130 , 0 , twelite.getTemperature(PAL1) , COLOR_YELLOW , COLOR_BLACK) ;
        x = disp7seg.dispCompensate(x+16 , 130 , 1 , twelite.getHumidity(PAL1) , COLOR_YELLOW , COLOR_BLACK) ;
  }
  if (envNo == 1) {
    sprintf(str,"%s 照度%5dLx",property.getPalName(2),twelite.getBrightness(PAL2)) ;
            fontSD.dispTextG(308, 166 , 320 , str , COLOR_GREEN) ; 
    // 補正済み環境値表示
    int x = disp7seg.dispCompensate(308, 186 , 0 , twelite.getTemperature(PAL2) , COLOR_GREEN , COLOR_BLACK) ;
        x = disp7seg.dispCompensate(x+16 , 186 , 1 , twelite.getHumidity(PAL2) , COLOR_GREEN , COLOR_BLACK) ;
  }
}

/**
 * @brief 気圧データ追加
 * @param pres 気圧（hPa）
 */
void addAirPressure(float pres) {
  airPress[airPressTop] = (int)((pres+0.05) * 10)  ;
  if (airPressNum < airPressMax) {
    airPressNum++ ; 
  }
  airPressTop = (airPressTop + 1) % airPressMax ;
}

/**
 * @brief 気圧グラフ表示
 * 
 */
void dispAirPressure() {
  if (controlMode != 0) return ;

  // --- 最大値・最小値を求める
  int maxPres = 0 ;
  int minPres = 0 ;
  if (airPressNum == 0) {
    maxPres = 10100 ;
    minPres = 10000 ;
  }
  
  for (int i=0;i<airPressNum;i++) {
    if (maxPres == 0 || maxPres < airPress[i]) { maxPres = airPress[i] ; }
    if (minPres == 0 || minPres > airPress[i]) { minPres = airPress[i] ; }
  }
  minPres = (minPres / 10) * 10 ; // 下位10の倍数に丸める
  maxPres = ((maxPres+9) / 10) * 10 ; // 上位10の倍数に丸める
  if ((maxPres-minPres) < 10) { maxPres = minPres + 10 ; }

  float pressStep = 79. ;
  if ((maxPres - minPres) > 0) {
    pressStep = 79. / (maxPres - minPres) ;
  }

  // --- グラフ表示
  sprintf(str,"%4d",maxPres/10) ;
  fontSD.dispTextG(308 , 230 , 32 , str , COLOR_WHITE ,false) ; 
  sprintf(str,"%4d",minPres/10) ;
  fontSD.dispTextG(308 , 304 , 32 , str , COLOR_WHITE ,false) ; 

  //--- グラフ描画（高速化: 背景を一度に塗り、緑部分のみ描画）
  tft.fillBox(344, 239, 135, 79, COLOR_DARK_GRAY); // 背景全体を黄色で塗る
  for (int i=0;i<airPressMax;i++) {
    int pos = (airPressTop + i) % airPressMax ;
    int h = (airPress[pos] - minPres) * pressStep ;
    if (h>79) h = 79 ;
    if (h > 0) { // 高さが0の場合は描画しない
      int x = 344 + airPressWide * i ;
      tft.fillBox(x , 239 + (79 - h), 7 , h , COLOR_WHITE) ; // 上から緑を描画
    }
  }
}
/**
 * @brief 画面初期表示
 * 
 */
void initDisp( ) {
  tft.fillScreen(COLOR_BLACK) ;

  disp7seg.dispIcon(328, 4, CONFIG_ICON , COLOR_GRAY, COLOR_BLACK) ;
  disp7seg.dispIcon(384, 4, ALARM_ICON  , (property.isAramEnabled() ? COLOR_WHITE : COLOR_GRAY), COLOR_BLACK) ;
  disp7seg.dispIcon(440, 4, LAUNDRY_ICON, COLOR_GRAY, COLOR_BLACK) ;

  dispWather( ) ;
  dispBME280(0.0,0.0,0.0) ;
  dispTWELITE(0) ;
  dispTWELITE(1) ;

  tft.drowBitmap(316 , 256 , 16 , 32 , disp7seg.disp7SEG(17 , COLOR_WHITE, COLOR_BLACK)) ;
  tft.fillBox(344, 238, 135, 81, COLOR_DARK_GRAY) ;
  tft.fillBox(344, 238, 135, 1 , COLOR_WHITE) ;
  tft.fillBox(344, 319, 135, 1 , COLOR_WHITE) ;

  dispAirPressure( ) ;
}

// ▲▼▲▼▲▼▲▼▲▼▲▼▲▼▲▼▲▼▲▼
void setup() {
  DEBUG_SERIAL_BEGIN(115200);
  delay(2000);

  initSDCard();
	tft.initTFT();
  tft.fillScreen(COLOR_WHITE);   
  aram.init( );

  /** 起動画像 */
  tft.readBMP(0,0,"/IMG/IMG.bmp");
  tft.setDir(3);
  
  // --- プロパティ読み込み
  if (!property.LoadProperty( )) {
    DPRINTLN("Property Load ERROR.") ;
    disp7seg.dispIcon(16,280,CONFIG_ICON,COLOR_RED,COLOR_WHITE) ;
    while(true) ;
  }
  disp7seg.dispIcon(16,280,CONFIG_ICON,COLOR_BLUE,COLOR_WHITE) ;
  init_TSC() ;

  xTaskCreatePinnedToCore(
    AramTask,
    "AramTask",
    2048,              // スタック量
    NULL,
    2,                 // 優先度
    &AramTaskHandle,
    0                  // Core0（WiFi）
  );
  
  // --- Netwaorkに接続
  DPRINTLN("CONNECT Wi-Fi ... ");
  disp7seg.dispIcon(56,280,WIFI_ICON,COLOR_GREEN,COLOR_WHITE) ;
  WiFi.begin(property.getWifiAP( ) , property.getWifiPWD( ));
  delay(1000*2) ;
  while (WiFi.status() != WL_CONNECTED) {
    DPRINTLN("connectWiFi(2) - false") ;
    disp7seg.dispIcon(56,280,WIFI_ICON,blinkIconColor(),COLOR_WHITE) ;
    WiFi.disconnect();
    delay(1000*2) ;
    disp7seg.dispIcon(56,280,WIFI_ICON,blinkIconColor(),COLOR_WHITE) ;
    WiFi.begin(property.getWifiAP( ) , property.getWifiPWD( ));
    delay(1000*2) ;
  }
  disp7seg.dispIcon(56,280,WIFI_ICON,blinkIconColor(),COLOR_WHITE) ;

  DPRINTLN("WiFi connected");
  DPRINT("IP address: ");
  DPRINTLN(WiFi.localIP());  

  // --- NTP & 天気予報取得
  flagNTP = false ;
  flagWATHER = false ;
  while (!jmaWather.getNTP()) {
    disp7seg.dispIcon(56,280,WIFI_ICON,blinkIconColor(),COLOR_WHITE) ;
    DPRINTLN("NTP CONNECT FALE");
    delay(1000*2) ;
  }
  while (!jmaWather.getWatherData( )) {
    disp7seg.dispIcon(56,280,WIFI_ICON,blinkIconColor(),COLOR_WHITE) ;
    DPRINTLN("HTTP REQUEST FALE");
    delay(1000*2) ;
  }
  disp7seg.dispIcon(56,280,WIFI_ICON,COLOR_BLUE,COLOR_WHITE) ;

  // --- BME280 & TWELITE 初期化
  bme.initBME280( ) ;
  twelite.init( ) ;

  for (int i=0;i<airPressMax;i++) {
    airPress[i] = 0;
  }
  initDisp( ) ;
}

// ▲▼▲▼▲▼▲▼▲▼▲▼▲▼▲▼▲▼▲▼
void loop() {
  // --- Touch Check ---
  if (isTouch( )) {
    uint16_t x , y ;
    if (readTouch(x,y)) {
      if (controlMode == 0) {
        // === 天気予報表示モード
        if (328<=x&&x<360 && 4<=y&&y<44) {
          // 設定アイコンタッチ
          controlMode = 1 ; // 都道府県選択モードへ
          DPRINTLN("設定アイコンタッチ");
          AreaCodeSel.setRegionCode(property.getRegionCode()) ;
          AreaCodeSel.setAreaCode(property.getAreaCode( )) ;
          AreaCodeSel.setCityName(property.getCityName( )) ;

          AreaCodeSel.loadCentersJSON( ) ;
          AreaCodeSel.makeMenuList("") ;
          AreaCodeSel.getDefMenuCur(1) ;
          AreaCodeSel.dispMenu();
        } else
        if (384<=x&&x<416 && 4<=y&&y<44) {
          // 設定アイコンタッチ
          controlMode = 11; // チャイム設定
          Serial.println("タイマーアイコンタッチ");
          for (int i=0;i<5;i++) {
            aram.setAramEnable(i, property.getAramEnable(i));
            aram.setAramTimeHour(i, property.getAramTimeHour(i));
            aram.setAramTimeMin(i, property.getAramTimeMin(i));
            aram.setAramMelody(i, property.getAramMelody(i));
            for (int d = 0; d < 7; d++) {
              aram.setAramWDay(i,d,property.getAramWDay(i,d)) ;
            }
          }
          aram.dispAramList( ) ;
        } else
        if (440<=x&&x<472 && 4<=y&&y<44) {
          // 設定アイコンタッチ
          // controlMode = 21 ; // レインセンサー　ON/OFF
          Serial.println("雨降りアイコンタッチ");
          if (AramTaskHandle) {
            Serial.println("ARAM TASK");
            AramTaskMode = 4 ; // 雨降り
            xTaskNotifyGive(AramTaskHandle);
          }
        }
      } else
      if (controlMode == 1) {
        // ---  天気予報：CONFIG:地域選択
        int sel = AreaCodeSel.checkMenuTouch(x,y) ;
        if (sel == 1 || sel == -1) {
          DPRINTF("AreaCodeSel Touch sel=%d\n",sel);
          AreaCodeSel.moveCursor(sel) ;
        } else
        if (sel == 2) { // 決定
          String code = AreaCodeSel.getCode(controlMode) ;
          DPRINTF("Selected Code=%s\n",code.c_str());
          // エリア選択完了→都道府県選択へ
          controlMode = 2 ;
          AreaCodeSel.makeMenuList(code) ;
          AreaCodeSel.getDefMenuCur(2) ;
          AreaCodeSel.dispMenu( ) ;
        } else
        if (sel == 3) { // 戻る
          // 戻る
          DPRINTF("AreaCodeSel Touch 戻る");
          // エリア選択→天気予報表示へ
          AreaCodeSel.clear( ) ;
          controlMode = 0 ;
          initDisp( ) ;
        }
      } else
      if (controlMode == 2) {
        // ---  天気予報：CONFIG:都道府県選択
        int sel = AreaCodeSel.checkMenuTouch(x,y) ;
        if (sel == 1 || sel == -1) {
          DPRINTF("AreaCodeSel Touch sel=%d\n",sel);
          AreaCodeSel.moveCursor(sel);
        } else
        if (sel == 2) { // 決定
          String code = AreaCodeSel.getCode(controlMode) ;
          controlMode = 3;
          DPRINTF("Selected Code=%s\n",code.c_str());
          AreaCodeSel.makeCityMenuList(code) ;
          AreaCodeSel.getDefMenuCur(3) ;
          AreaCodeSel.dispMenu();
        } else
        if (sel == 3) { // 戻る
          // 戻る
          DPRINTF("AreaCodeSel Touch 戻る");
          // 都道府県選択→エリア選択へ
          controlMode = 1 ;
          AreaCodeSel.makeMenuList("") ;
          AreaCodeSel.getDefMenuCur(1) ;
          AreaCodeSel.dispMenu( ) ;
        }
      } else
      if (controlMode == 3) {
        // ---  天気予報：CONFIG:市町村選択
        int sel = AreaCodeSel.checkMenuTouch(x,y) ;
        if (sel == 1 || sel == -1) {
          DPRINTF("AreaCodeSel Touch sel=%d\n",sel);
          AreaCodeSel.moveCursor(sel);
        } else
        if (sel == 2) { // 決定
          String code = AreaCodeSel.getCode(controlMode) ;
          DPRINTF("regionCode=%s\n",AreaCodeSel.getRegionCode().c_str()) ;
          DPRINTF("regionName=%s\n",AreaCodeSel.getRegionName().c_str()) ;
          DPRINTF("areaCode=%s\n",AreaCodeSel.getAreaCode().c_str()) ;
          DPRINTF("CityName=%s\n",AreaCodeSel.getCityName().c_str()) ;

          property.setRegionCode(AreaCodeSel.getRegionCode()) ;
          property.setAreaCode(AreaCodeSel.getAreaCode()) ;
          property.setRegionName(AreaCodeSel.getRegionName()) ;
          property.setCityName(AreaCodeSel.getCityName()) ;
          property.SaveProperty( ) ;
          AreaCodeSel.clear( ) ;

          // 市町村選択完了→天気予報表示へ
          controlMode = 0 ;
          jmaWather.getWatherData( ) ;
          initDisp( ) ;
        } else
        if (sel == 3) { // 戻る
          // 戻る
          DPRINTF("AreaCodeSel Touch 戻る");
          // 市町村選択→都道府県選択へ
          AreaCodeSel.loadCentersJSON( ) ;
          DPRINTF("loadCentersJSON() END") ;
          controlMode = 2 ;
          String code = AreaCodeSel.getCenterCode( ) ;
          DPRINTF("Selected Code=%s\n",code.c_str());
          AreaCodeSel.makeMenuList(code) ;
          AreaCodeSel.getDefMenuCur(2) ;
          AreaCodeSel.dispMenu( ) ;
        }
      } else
      if (controlMode == 11) {
        int no = aram.checkMenuTouch(x,y) ;
        Serial.printf("ARAM Touch sel=%d\n",no);
        if (no >= 1 && no <= 5) {
          // アラーム設定選択
          aramSel = no ;
          Serial.printf("Selected ARAM No=%d\n",aramSel);
          // ---  ARAM：CONFIG:ALARM SETTING
          aram.copyAramToWork(aramSel-1) ;
          aram.dispEditAram() ;
          controlMode = 12 ;
        } else
        if (no == 10) {
          // 戻る
          DPRINTF("ARAM Touch 戻る"); 
          controlMode = 0 ;
          aramSel = 0 ;
          initDisp( ) ;
        } else
        if (no == 20) {
          // 決定
          DPRINTF("ARAM Touch 決定"); 
          for (int i=0;i<5;i++) {
            property.setAramEnable(i, aram.getAramEnable(i));
            property.setAramTimeHour(i, aram.getAramTimeHour(i));
            property.setAramTimeMin(i, aram.getAramTimeMin(i));
            property.setAramMelody(i, aram.getAramMelody(i));
            for (int d = 0; d < 7; d++) {
              property.setAramWDay(i,d,aram.getAramWDay(i,d)) ;
            }
          }
          property.SaveProperty( ) ;
          controlMode = 0 ;
          aramSel = 0 ;
          initDisp( ) ;
        }
      } else
      if (controlMode == 12) {
        // ---  ARAM：CONFIG:ALARM EDIT
        int sel = aram.checkEditMenuTouch(x,y) ;
        Serial.printf("ARAM Edit Touch sel=%d\n",sel);
        if (sel == 20) { // 決定
          Serial.println("ARAM Edit 決定");
          aram.copyWorkToAram(aramSel-1) ;
          controlMode = 11 ;
          aramSel = 0 ;
          aram.dispAramList( ) ;
        } else
        if (sel == 10) { // 戻る
          // 戻る
          Serial.println("ARAM Edit 戻る");
          controlMode = 11 ;
          aramSel = 0 ;
          aram.dispAramList( ) ;
        }
      }
      // ---      
    }
  }

  // --- Disp Clock ---
  struct tm timeinfo;
  if (!getLocalTime(&timeinfo)) {
    DPRINTLN("Failed to obtain time");
    return;
  }

  // controlMode 0:天気予報表示
  if (timeinfo.tm_sec != lastSec) {
    lastSec = timeinfo.tm_sec ;
    if (controlMode == 0) {
      disp7seg.dispTime(2, 2 ,timeinfo.tm_year + 1900 , timeinfo.tm_mon+1 , timeinfo.tm_mday , timeinfo.tm_hour , timeinfo.tm_min ,timeinfo.tm_sec , COLOR_WHITE , COLOR_BLACK) ; 
    }
    if (timeinfo.tm_sec % 10 == 0) {
      // BME280 の補正済み環境値: 温度（℃）, 気圧（hPa）, 湿度（%）
      float temp ;
      float pres ;
      float hum ;
      bme.getCompensate(temp,pres,hum) ;
      dispBME280(temp,pres,hum) ;

      if (airPressNum < airPressMax || (timeinfo.tm_sec == 0 && timeinfo.tm_min % 5 == 0)) {
        addAirPressure(pres);
        dispAirPressure( ) ;
      }
    }

    if (timeinfo.tm_hour != jmaWather.ntpTime.tm_hour) {
      DPRINTLN("CHECK TIME");
      if (flagNTP) {
        DPRINTLN("flagNTP TRUE");
        flagNTP = false ;
      } else {
        DPRINTLN("flagNTP FALSE");
        jmaWather.getNTP() ;
        flagNTP = (timeinfo.tm_hour == jmaWather.ntpTime.tm_hour) ;
        jmaWather.ntpTime = timeinfo ;
        flagWATHER = (timeinfo.tm_hour == jmaWather.ntpTime.tm_hour) ;
        DPRINTLN((flagNTP?"flagNTP=TRUE":"flagNTP=FALSE"));
        DPRINTF("NTP: %02d:%02d:%02d\n",timeinfo.tm_hour , timeinfo.tm_min ,timeinfo.tm_sec) ;
        DPRINTF("JAM: %02d:%02d:%02d\n",jmaWather.ntpTime.tm_hour , jmaWather.ntpTime.tm_min) ;
      }
    } else {
      if (flagWATHER) {
        DPRINTLN("CHECK WATHER");
        flagWATHER = false;
        jmaWather.getWatherData( ) ;
        dispWather( );          
      }
    }

    // アラームチェック
    if (property.isAramEnabled()) {
//      DPRINTF("アラームチェック: timeinfo.tm_wday=%d\n",timeinfo.tm_wday) ;
      for (int i=0;i<5;i++) {
//        DPRINTF("ARAM TIME No=%d  getAramWDay(%d,%d)=%d\n",i,i,timeinfo.tm_wday,property.getAramWDay(i,timeinfo.tm_wday));
        if (property.getAramTimeMin(i) == timeinfo.tm_min &&
            property.getAramTimeHour(i) == timeinfo.tm_hour &&
            property.getAramWDay(i,timeinfo.tm_wday) == 1 &&
            timeinfo.tm_sec == 0 &&
            property.getAramEnable(i)) {
          DPRINTF("ARAM TIME No=%d\n",i+1);
          if (AramTaskHandle) {
            Serial.println("ARAM TASK");
            AramTaskMode = property.getAramMelody(i) ;
            xTaskNotifyGive(AramTaskHandle);
          }
        }
      }
    }
  }

  // TWELITE 環境センサー値取得
  int envNo = twelite.readSerial( ) ;
  if (envNo !=-1) {
    dispTWELITE(envNo) ;
  }

  delay(100) ;
}
