/**
 * プロパティ情報の LOAD/SAVE
 */
#include <ArduinoJson.h>
#include <SD.h>
#include "property.h"
#include "debug.h"
Property property ;

/** コンストラクタ */
Property::Property( ) {
  pal0Name = PAN0NAME;    // センサー配置名称 0
  pal1Name = PAN1NAME;    // センサー配置名称 1
  pal2Name = PAN2NAME;    // センサー配置名称 2
  for (int i=0;i<5;i++) {
    aram[i].enable = 0 ;
    aram[i].timehour = 0 ;
    aram[i].timemin = 0 ;
    aram[i].melody = 0 ;
  }
}

/** センサー配置名称取得 */
const char* Property::getPalName(int no) {
  if (no == 0) { return pal0Name.c_str();}
  if (no == 1) { return pal1Name.c_str();}
  if (no == 2) { return pal2Name.c_str();}
  return "---" ;
}

/** プロパティ情報のロード */
bool Property::LoadProperty( ) {
  DPRINTLN("LoadProperty( )") ;
  if (!SD.exists(property_FILE_NAME)) {
      DPRINTF("File %s not exists.\n",property_FILE_NAME) ;
      return false ;
  }  
  // --- File Load ---
  File file = SD.open(property_FILE_NAME);
  if (!file) {
    DPRINTF("File %s Open Error.\n",property_FILE_NAME) ;
    return false ;
  }

  JsonDocument doc;
  DeserializationError error = deserializeJson(doc, file);
  file.close();
  if (error) {
    DPRINT(F("deserializeJson() failed: "));
    DPRINTLN(error.f_str());
    return false ;
  }

  String appName = doc["APLName"] ;
  String appVer = doc["Ver"] ;
  if (appName != "WeatherForecast" || appVer != "2") {
    return false ;
  }
  
  wifiAP = doc["WiFi"]["SSID"].as<String>() ;           // Wi-Fi アクセスポイント
  wifiPWD = doc["WiFi"]["PWD"].as<String>() ;           // Wi-Fi パスワード
  weatherURL = doc["Forecast"]["URL"].as<String>() ;    // 天気予報取得用URL
  regionCode = doc["Forecast"]["RCode"].as<String>() ;  // 地域コード
  areaCode = doc["Forecast"]["ACode"].as<int>() ;       // 予報区コード
  regionName = doc["Forecast"]["RName"].as<String>() ;  // 地域コード
  cityName = doc["Forecast"]["CName"].as<String>() ;    // 予報区コード

  // ---- Alarm ----
  JsonArray alarmArray = doc["ARAM"];
  for (int i = 0; i < 5; i++) {
    aram[i].enable   = alarmArray[i]["enable"]|0;
    aram[i].timehour = alarmArray[i]["hour"]|0;
    aram[i].timemin  = alarmArray[i]["min"]|0;
    aram[i].melody   = alarmArray[i]["melody"]|0;
    
    for (int d = 0; d < 7; d++) {
      aram[i].wday[d] = alarmArray[i]["wday"][d]|0;
    }
  }

  DPRINTLN("--- PROPERTY DATA");
  DPRINTF("appName=%s\n",appName.c_str());
  DPRINTF("appVer=%s\n",appVer.c_str());
//  DPRINTLN(wifiAP) ;
//  DPRINTLN(wifiPWD) ;
  DPRINTF("weatherURL=%s\n",weatherURL.c_str());
  DPRINTF("regionCode=%s\n",regionCode.c_str());
  DPRINTF("areaCode=%d\n",areaCode);
  DPRINTF("regionName=%s\n",regionName.c_str());
  DPRINTF("cityName=%s\n",cityName.c_str());
  for (int i=0;i<5;i++) {
    DPRINTF("ARAM[%d] enable=%d  hour=%02d  min=%02d melody=%d\n",
                  i,aram[i].enable,  aram[i].timehour ,aram[i].timemin ,aram[i].melody ) ;
    DPRINT("WDAY:") ;
    for (int d = 0; d < 7; d++) {
      DPRINTF(" %d",aram[i].wday[d]) ;
    }
    DPRINTLN() ;
  }

  return true ;
}

/** プロパティ情報のセーブ */
bool Property::SaveProperty( ) {

  StaticJsonDocument<2048> doc;

  // ---- Application info ----
  doc["APLName"] = "WeatherForecast";
  doc["Ver"]     = "2";

  // ---- WiFi ----
  JsonObject wifi = doc.createNestedObject("WiFi");
  wifi["SSID"] = wifiAP;
  wifi["PWD"]  = wifiPWD;

  // ---- Forecast ----
  JsonObject forecast = doc.createNestedObject("Forecast");
  forecast["URL"]   = weatherURL;
  forecast["RCode"] = regionCode;
  forecast["RName"] = regionName;
  forecast["ACode"] = areaCode;
  forecast["CName"] = cityName;

  // ---- ARAM ----
  JsonArray alarm = doc.createNestedArray("ARAM");
  for (int i = 0; i < 5; i++) {
    JsonObject a = alarm.createNestedObject();

    a["enable"] = aram[i].enable;
    a["hour"]   = aram[i].timehour;
    a["min"]    = aram[i].timemin;
    a["melody"] = aram[i].melody;

    JsonArray wday = a.createNestedArray("wday");
    for (int d = 0; d < 7; d++) {
      wday.add(aram[i].wday[d]);
    }
  }

  // 上書き保存
  File file = SD.open(property_FILE_NAME, "w");
  if (!file) {
    DPRINTLN("ファイルの書き込みに失敗しました");
    return false ;
  }

  serializeJsonPretty(doc, file);  // 整形して保存
  file.flush();  // 🔴 ここでバッファの内容を確実にSDカードに書き込み
  file.close();

  DPRINTLN("JSONファイルを更新・保存しました");

  return true ;
}

bool Property::isAramEnabled( ) {
  for (int i=0;i<5;i++) {
    if (aram[i].enable != 0) {
      return true ;
    }
  }
  return false ;
}
